/**
 * ts-release.js
 * Copyright (c)2000 - 2017 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * Sungmin Kim <sm.art.kim@samsung.com>
 * Jonghwan Park <iwin100.park@samsung.com>
 * Kitae Kim <kt920.kim@samsung.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
**/

'use strict';

var async = require('async');
var path = require('path');
var _ = require('underscore');

var TsRepo = require('../../org.tizen.ts.cli.repository/ts-repo.js');
var release = require('../../org.tizen.ts.cli.release/release.js');

module.exports.createInstaller = function (argv, log) {
    var repoLocation = argv.r;
    var metaPkgList = argv.M;
    var excludePkgList = argv.X;
    var baseRepoUrl = argv.u;
    var title = argv.t;
    var outputName = argv.O || null;
    var baseSnapshot = argv['base-snapshot'] || null;

    if (!repoLocation) {
        log.error('Required --repository option!');
        var message = 'Usage: ts-cli create-installer --repository <repository path> ' +
            '--meta <meta package list> [--url <base repository URL>] ' +
            '[--exclude-meta <meta package list>] [--title <Installer title>]';
        log.error(message);
        process.exit(-1);
    }

    var distName = path.basename(repoLocation);
    var targetDist = null;

    log.info('Tizen Studio Installer Creation Process - Start');

    var targetRepo = new TsRepo({
        location: repoLocation
    });

    async.waterfall([
        function (cb) {
            // open repository
            log.info(' # - Open repostiory ' + repoLocation);
            targetRepo.open(function (err) {
                if (!err) {
                    targetDist = targetRepo.distributions[distName];
                    if (!targetDist) {
                        err = new Error('# - Failed to get \'' + distName + '\'');
                    }
                } else {
                    log.error(err);
                }
                cb(err);
            });
        },
        function (cb) {
            var options = {
                log: log,
                baseRepoUrl: baseRepoUrl,
                metaPkgList: metaPkgList,
                excludePkgList: excludePkgList,
                installerTitle: title,
                outputName: outputName,
                baseSnapshot: baseSnapshot
            };

            release.createInstaller(targetDist, options, cb);
        }
    ],
    function (err) {
        if (err) {
            log.error('Creating an Installer - Failure!');
            log.error(err);
            process.exit(-1);
        }
        log.info('Creating an Installer - Success');
        process.exit(0);
    });
};


module.exports.createImage = function (argv, log) {
    var repoLocation = argv.r;
    var excludePkgList = argv.X;
    var baseRepoUrl = argv.u;
    var skipIntegrityCheck = argv.S;
    var outputName = argv.O || null;
    var targetOS = argv.os || null;
    var baseSnapshot = argv['base-snapshot'] || null;

    if (!repoLocation) {
        log.error('Required --repository option!');
        log.error('Usage: ts-cli create-image --repository <repository path> [--url <base repository URL>] [--exclude-meta <meta package list>] [--os <os name>] [--base-snapshot <base snapshot>]');
        process.exit(-1);
    }

    var distName = path.basename(repoLocation);
    var targetDist = null;

    log.info('Tizen Studio Image Creation Process - Start');

    var targetRepo = new TsRepo({
        location: repoLocation
    });

    async.series([
        function (cb) {
            // open repository
            log.info(' # - Open repostiory ' + repoLocation);
            targetRepo.open(function (err) {
                if (!err) {
                    targetDist = targetRepo.distributions[distName];
                    if (!targetDist) {
                        err = new Error('# - Failed to get \'' + distName + '\'');
                    }
                } else {
                    log.error(err);
                }
                cb(err);
            });
        },
        function (cb) {
            var options = {
                log: log,
                baseRepoUrl: baseRepoUrl,
                targetOS: targetOS,
                excludePkgList: excludePkgList,
                skipIntegrityCheck: skipIntegrityCheck,
                outputName: outputName,
                baseSnapshot: baseSnapshot
            };

            release.createImage(targetDist, options, cb);
        }
    ],
    function (err) {
        if (err) {
            log.error('Creating an Image - Failure!');
            log.error(err);
            process.exit(-1);
        }
        log.info('Creating an Image - Success');
        process.exit(0);
    });
};

