/**
 * package.js
 * Copyright (c) 2000 - 2017 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * Sungmin Kim <sm.art.kim@samsung.com>
 * Jonghwan Park <iwin100.park@samsung.com>
 * Kitae Kim <kt920.kim@samsung.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
**/

'use strict';

var _ = require('underscore');

module.exports = Package;

function Package(object) {
    this.name = object.Package;
    this.label = object.Label;
    this.version = object.Version;
    this.os = object.OS[0];
    this.osList = object.OS;
    this.buildHostOS = object['Build-host-os'];
    this.maintainer = object.Maintainer;
    this.attr = object.Attribute;
    this.installDepList = (object['Install-dependency'] === undefined) ? [] : object['Install-dependency'];
    this.buildDepList = (object['Build-dependency'] === undefined) ? [] : object['Build-dependency'];
    this.srcDepList = (object['Source-dependency'] === undefined) ? [] : object['Source-dependency'];
    this.conflicts = object.Conflicts;
    this.source = object.Source;
    this.srcPath = object['Src-path'];
    this.path = object.Path;
    this.origin = object.Origin;
    this.checksum = object.SHA256;
    this.size = parseInt(object.Size, 10);
    this.uncompSize = object['Uncompressed-size'];
    this.desc = object.Description;
    this.custom = getCustom(object);
    this.changelog = object.Changelog;
}

Package.prototype.toString = toString;

function getCustom(object) {
    var customList = _.select(_.keys(object), function (key) {
        return (key.substring(0, 2) === 'C-');
    });
    var result = {};
    _.each(customList, function (custom) {
        result[custom] = object[custom];
    });
    return result;
}

function depString(dep) {
    var string = dep.packageName;
    if (dep.comp) {
        string = string + ' (' + dep.comp + ' ' + dep.baseVersion + ')';
    }
    if (dep.os) {
        string = string + ' [' + dep.os + ']';
    }
    return string;
}

function depListString(depList) {
    return _.map(depList, function (dep) {
        return depString(dep);
    }).join(', ');
}

function toString() {
    var pkg = this;

    var string = 'Package : ' + pkg.name;

    if (pkg.label) {
        string += '\nLabel : ' + pkg.label;
    }
    if (pkg.version) {
        string += '\nVersion : ' + pkg.version;
    }
    if (pkg.osList) {
        string += '\nOS : ' + pkg.osList.join(', ');
    }
    if (pkg.buildHostOS) {
        string += '\nBuild-host-os : ' + pkg.buildHostOS;
    }
    if (pkg.maintainer) {
        string += '\nMaintainer : ' + pkg.maintainer.join(', ');
    }
    if (pkg.attr) {
        string += '\nAttribute : ' + pkg.attr;
    }
    if (pkg.installDepList.length > 0) {
        string += '\nInstall-dependency : ' + depListString(pkg.installDepList);
    }
    if (pkg.buildDepList.length > 0) {
        string += '\nBuild-dependency : ' + depListString(pkg.buildDepList);
    }
    if (pkg.srcDepList.length > 0) {
        string += '\nSource-dependency : ' + depListString(pkg.srcDepList);
    }
    if (pkg.conflicts) {
        string += '\nConflicts : ' + pkg.conflicts;
    }
    if (pkg.source) {
        string += '\nSource : ' + pkg.source;
    }
    if (pkg.srcPath) {
        string += '\nSrc-path : ' + pkg.srcPath;
    }
    if (pkg.path) {
        string += '\nPath : ' + pkg.path;
    }
    if (pkg.origin) {
        string += '\nOrigin : ' + pkg.origin;
    }
    if (pkg.checksum) {
        string += '\nSHA256 : ' + pkg.checksum;
    }
    if (pkg.size) {
        string += '\nSize : ' + pkg.size;
    }
    if (pkg.uncompSize) {
        string += '\nUncompressed-size : ' + pkg.uncompSize;
    }
    _.each(_.keys(pkg.custom), function (key) {
        string += '\n' + key + ' : ' + pkg.custom[key].replace(/\n\n/g, '\n.\n').replace(/\n/g, '\n ');
    });
    if (pkg.desc) {
        string += '\nDescription : ' + pkg.desc.replace(/\n\n/g, '\n.\n').replace(/\n/g, '\n ');
    }
    string += '\n';
    return string;
}

