/**
 * plugin.js
 * Copyright (c) 2000 - 2015 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * DongHee Yang <donghee.yang@samsung.com>
 * Sungmin Kim <sm.art.kim@samsung.com>
 * Jiil Hyoun <jiil.hyoun@samsung.com>
 * Jonghwan Park <iwin100.park@samsung.com>
 * Kitae Kim <kt920.kim@samsung.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
**/

var path = require('path');
var fs = require('fs');

var Extension = require('./extension.js');
var ExtensionPoint = require('./extension-point.js');

/**
 * @module core/plugin/plugin
 */

/**
 * @constructor
 * @param {string} json
 * @param {string} jpath
 * @memberOf module:core/plugin/plugin
 */
function Plugin(json, jpath) {
    var self = this;

    /** @type {string} */
    this.name = json.name;
    if (json.type !== undefined) {
        /** @type {string} */
        this.type = json.type;
    } else {
        this.type = 'core';
    }

    if (json.description !== undefined) {
        /** @type {string} */
        this.description = json.description;
    } else {
        this.description = '';
    }

    if (json.version !== undefined) {
        /** @type {string} */
        this.version = json.version;
    } else {
        this.version = '1.0.0';
    }

    /** @type {Array.<dep>} */
    this.deps = [];
    if (json.deps !== undefined) {
        for (var i = 0; i < json.deps.length; i++) {
            var dep = json.deps[i];

            this.deps.push(dep);
        }
    }

    /** @type {Array.<module:core/plugin/extension-point.ExtensionPoint>} */
    this.extensionPoints = [];
    if (json.extensionPoints != undefined) {
        for (var i = 0; i < json.extensionPoints.length; i++) {
            var exp = ExtensionPoint.create(json.extensionPoints[i], this);

            if (exp != null) {
                this.extensionPoints.push(exp);
            }
        }
    }

    /** @type {Array.<module:core/plugin/extension.Extension>} */
    this.extensions = [];
    if (json.extensions != undefined) {
        for (var i = 0; i < json.extensions.length; i++) {
            var ex = Extension.create(json.extensions[i], this);

            if (ex != null) {
                this.extensions.push(ex);
            }
        }
    }


    /** @type {string} */
    this.pluginPath = path.dirname(jpath);


    /**
     * @method load
     * @returns {object} - plugin module
     * @memberOf module:core/plugin/plugin.Plugin
     */
    this.load = function () {
        var pluginInitPath = path.join(self.pluginPath, 'activator.js');
        if (fs.existsSync(pluginInitPath)) {
            return require(pluginInitPath);
        } else {
            return {
                definition: self
            };
        }
    };
}


/**
 * @function create
 * @param {string} jsonPath - jsonPath
 * @returns {module:core/plugin/plugin.Plugin}
 * @memberOf module:core/plugin/plugin
 */

module.exports.create = function (jsonPath) {
    var j = require(jsonPath);

    // verify json format
    if (j.name === undefined || typeof j.name !== 'string') {
        return null;
    }
    if (j.description !== undefined && typeof j.description !== 'string') {
        return null;
    }
    if (j.version !== undefined && typeof j.version !== 'string') {
        return null;
    }
    if (j.disable === true) {
        return null;
    }


    return new Plugin(j, jsonPath);
};

