/**
 * project.js
 * Copyright (c) 2000 - 2015 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * DongHee Yang <donghee.yang@samsung.com>
 * Sungmin Kim <sm.art.kim@samsung.com>
 * Jiil Hyoun <jiil.hyoun@samsung.com>
 * Jonghwan Park <iwin100.park@samsung.com>
 * Kitae Kim <kt920.kim@samsung.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
**/

var dibs = require('../../../core/dibs.js');
var async = require('async');
var expressio = require('express.io');
var session = require('./session');
var _ = require('underscore');

var DError = require('../../../core/exception.js');

var router = new expressio.Router();

module.exports.router = router;

router.get('/projects', session.checkSession, function (req, res) {

    var email = req.session.email;
    var distName = req.query.distName;

    async.waterfall([
        function (cb) {
            session.isAdmin(email, cb);
        },
        function (isAdmin, cb) {
            if (isAdmin) {
                cb(null);
            } else {
                dibs.rpc.datamgr.checkPrivilege('project', 'READ', {
                    distName: distName
                }, email, function (err, allowed) {
                    if (allowed) {
                        cb(null);
                    } else {
                        cb(new DError('SESSION006', err));
                    }
                });
            }
        }], function (err) {
        if (err) {
            dibs.log.error(JSON.stringify(err));
            res.status(503).send(JSON.stringify(err.message));
        } else {
            getProjectList(req, res);
        }
    });
});


function getProjectList(req, res) {
    //default
    var order = 'name';
    var arrange = 'ASC';

    var email = req.session.email;
    var cond = {};

    // set queryPrivilege option if exists
    var sortBy = req.query.sortBy;
    if (sortBy) {
        if (sortBy[0] === '-') {
            cond.arrange = 'DESC';
        } else {
            cond.arrange = 'ASC';
        }
        cond.order = sortBy.substring(1, sortBy.length);
    } else {
        cond.order = order;
        cond.arrange = arrange;
    }

    // set queryPrivilege option if exists
    var queryPrivilege = req.query.queryPrivilege;
    if (queryPrivilege) {
        cond.queryPrivilege = queryPrivilege;
        cond.email = email;
    }

    // set queryPrivilege option if exists
    var privilege = req.query.privilege;
    if (privilege) {
        cond.privilege = privilege;
        cond.email = email;
    }

    // set distribution name option if exists
    var distName = req.query.distName;
    if (distName) {
        cond.distName = distName;
    }

    // set distribution name option if exists
    var name = req.query.name;
    if (name) {
        cond.name = name;
    }

    // set project type option if exists
    var prjType = req.query.type;
    if (prjType) {
        cond.type = prjType;
    }

    dibs.rpc.datamgr.searchProjects(cond, function (err, projects) {
        if (err) {
            dibs.log.error(JSON.stringify(err));
            res.status(503).send(JSON.stringify(err.message));
        } else {
            var resultProjects = [];
            _.each(projects, function (project) {
                if (project.options) {
                    Object.keys(project.options).forEach(function (key) {
                        project[key] = project.options[key];
                    });
                }
                resultProjects.push(project);
            });

            var data = {
                identifier: 'id',
                label: 'name',
                items: resultProjects
            };
            res.send(data);
        }
    });
}


router.get('/admin/projects', session.checkSession, function (req, res) {
    var distName = req.query.distName;
    var email = req.session.email;

    async.waterfall([
        function (cb) {
            dibs.rpc.datamgr.checkPrivilege('project', 'READ', {
                distName: distName
            }, email, function (err, allowed) {
                if (allowed) {
                    cb(null);
                } else {
                    cb(new DError('SESSION006'), err);
                }
            });
        }], function (err) {
        if (err) {
            dibs.log.error(JSON.stringify(err));
            res.status(503).send(JSON.stringify(err.message));
        } else {
            getAdminProjectList(req, res);
        }
    });
});


function getAdminProjectList(req, res) {
    var options = {};

    // set distribution name option if exists
    var distName = req.query.distName;
    if (distName) {
        options.distName = distName;
    }

    dibs.rpc.datamgr.searchProjects(options, function (err, projects) {
        if (err) {
            dibs.log.error(JSON.stringify(err));
            res.status(503).send(JSON.stringify(err.message));
        } else {
            async.mapSeries(projects,
                function (project, mcb) {
                    dibs.rpc.datamgr.searchProjectPrivilege({
                        id: project.id
                    }, function (err, groups) {
                        if (err) {
                            groups = {};
                        }
                        project.groups = groups;
                        mcb(null);
                    });
                },
                function (err) {
                    if (err) {
                        res.status(503).send(JSON.stringify(err.message));
                    } else {
                        res.send(JSON.stringify(projects));
                    }
                });
        }
    });
}


router.get('/projects/:id', session.checkSession, function (req, res) {
    var email = req.session.email;
    var projectId = Number(req.param('id'));
    var project = null;

    if (!projectId) {
        res.status(503).send('project id is undefined!');
        return;
    }

    var cond = {
        id: projectId
    };

    async.waterfall([
        function (cb) {
            dibs.rpc.datamgr.searchProjects(cond, function (err, projects) {
                if (err) {
                    cb(err);
                } else if (!projects || projects.length <= 0) {
                    cb(new DError('WEB001', {
                        arguments: 'project ID: ' + projectId
                    }));
                } else {
                    cb(null, projects);
                }
            });
        },
        function (projects, cb) {
            project = projects[0];
            dibs.rpc.datamgr.checkPrivilege('project', 'READ', {
                distName: project.distName
            }, email, function (err, allowed) {
                if (allowed) {
                    cb(null);
                } else {
                    cb(new DError('SESSION006'), err);
                }
            });
        },
        function (cb) {
            dibs.rpc.datamgr.searchProjectPrivilege({
                project_id: project.id
            }, cb);
        }], function (err, groups) {
        if (err) {
            dibs.log.error(JSON.stringify(err));
            res.status(503).send(JSON.stringify(err.message));
        } else {
            if (!groups) {
                groups = {};
            }
            project.groups = groups;
            res.send(JSON.stringify(project));
        }
    });
});


router.post('/projects', session.checkSession, function (req, res) {
    var distName = req.body.distName;
    var email = req.session.email;

    async.waterfall([
        function (cb) {
            dibs.rpc.datamgr.checkPrivilege('project', 'CREATE', {
                distName: distName
            }, email, function (err, allowed) {
                if (allowed) {
                    cb(null);
                } else {
                    cb(new DError('SESSION006'), err);
                }
            });
        }], function (err) {
        if (err) {
            dibs.log.error(JSON.stringify(err));
            res.status(503).send(JSON.stringify(err.message));
        } else {
            createNewProject(req, res);
        }
    });
});


function createNewProject(req, res) {
    var name = req.body.name;
    var type = req.body.type;
    var options = req.body.options;
    var environments = req.body.environments;
    var triggers = req.body.triggers;
    var notifications = req.body.notifications;
    var distribution = req.body.distName;
    var groups = req.body.groups;

    dibs.rpc.datamgr.addProject(name, type, options, environments, triggers, notifications, distribution, function (err) {
        if (err) {
            dibs.log.error(err);
            res.status(503).send(JSON.stringify(err.message));
            return;
        }

        if (groups) {
            dibs.rpc.datamgr.searchProjects({
                name: name,
                distName: distribution
            }, function (err, projects) {
                if (err) {
                    dibs.log.error(err);
                    res.status(503).send(JSON.stringify(err.message));
                    return;
                }
                var project = projects[0];
                async.eachSeries(groups,
                    function (groupName, ecb) {
                        dibs.rpc.datamgr.searchUserGroups({
                            name: groupName
                        }, function (err, data) {
                            if (err) {
                                return ecb(err);
                            }
                            var group = data[0];
                            if (_.contains(group.projects, project.id)) {
                                ecb(null);
                            } else {
                                group.projects.push(project.id);
                                dibs.rpc.datamgr.updateUserGroup(group, function (err) {
                                    ecb(err);
                                });
                            }
                        });
                    },
                    function (err) {
                        if (err) {
                            dibs.log.error(err);
                            res.status(503).send(JSON.stringify(err.message));
                        } else {
                            res.send();
                        }
                    });
            });
        } else {
            res.send();
        }
    });
}


router.put('/projects', session.checkSession, function (req, res) {
    var email = req.session.email;
    var prjId = Number(req.body.id);
    var name = req.body.name;
    var type = req.body.type;
    var status = req.body.status;
    var options = req.body.options;
    var environments = req.body.environments ? req.body.environments : [];
    var triggers = req.body.triggers ? req.body.triggers : [];
    var notifications = req.body.notifications ? req.body.notifications : [];
    var groups = req.body.groups;

    async.waterfall([
        function (cb) {
            dibs.rpc.datamgr.searchProjects({
                id: prjId
            }, function (err, prjs) {
                if (err) {
                    cb(err, null);
                } else if (prjs.length < 1) {
                    cb(new Error('The project not found!'), null);
                } else {
                    cb(null, prjs[0]);
                }
            });
        },
        function (prj, cb) {
            dibs.rpc.datamgr.checkPrivilege('project', 'UPDATE', {
                distName: prj.distName
            }, email, function (err, allowed) {
                if (allowed) {
                    cb(null, prj);
                } else {
                    cb(new DError('SESSION006', err), null);
                }
            });
        },
        function (prj, cb) {
            prj.type = type;
            prj.name = name;
            prj.status = status;
            prj.environments = environments;
            prj.triggers = triggers;
            prj.notifications = notifications;
            prj.groups = groups;

            if (options) {
                if (!prj.options) {
                    prj.options = {};
                }
                for (var key in options) {
                    prj.options[key] = options[key];
                }
            }
            dibs.rpc.datamgr.updateProject(prj, function (err) {
                cb(err);
            });
        }], function (err) {
        if (err) {
            dibs.log.error(err);
            res.status(503).send(JSON.stringify(err.message));
        } else {
            res.send();
        }
    });
});


router.delete('/projects', session.checkSession, function (req, res) {
    var email = req.session.email;
    var projectId = req.body;
    if (projectId.length <= 0) {
        res.status(503).send(JSON.stringify('Need project id'));
        return;
    }
    async.eachSeries(projectId, function (id, ecb) {
        async.waterfall([
            function (cb) {
                dibs.rpc.datamgr.searchProjects({
                    id: id
                }, function (err, results) {
                    if (!err && results[0]) {
                        cb(null, results[0]);
                    } else {
                        dibs.log.error(err);
                        cb(new Error('Fail to find the project!'), null);
                    }
                });
            },
            function (prj, cb) {
                dibs.rpc.datamgr.checkPrivilege('project', 'DELETE', {
                    distName: prj.distName
                }, email, function (err, allowed) {
                    if (allowed) {
                        cb(null, prj);
                    } else {
                        cb(new DError('SESSION006', err), null);
                    }
                });
            },
            function (prj, cb) {
                dibs.rpc.datamgr.removeProject(prj.name, prj.distName, cb);
            }], function (err) {
            if (err) {
                dibs.log.error(err);
            }
            ecb(err);
        });
    },
        function (err) {
            if (err) {
                res.status(503).send(JSON.stringify(err.message));
            } else {
                res.send();
            }
        });
});
