/**
 * ts-project.js
 * Copyright (c)2000 - 2017 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * Sungmin Kim <sm.art.kim@samsung.com>
 * Jonghwan Park <iwin100.park@samsung.com>
 * Kitae Kim <kt920.kim@samsung.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
**/

var async = require('async');
var fs = require('fs');
var os = require('os');
var path = require('path');

var util = require('../org.tizen.ts.base.common/util.js');
var PkgInfo = require('../org.tizen.ts.base.common/pkg-info.js');
var TsSourceJob = require('./ts-source-job.js');
var TsBinaryJob = require('./ts-binary-job.js');
var TsArchiveJob = require('./ts-archive-job.js');

module.exports.create = function (targetPath, options, callback) {
    if (!targetPath) {
        targetPath = process.cwd;
    } else {
        if (path.isAbsolute(targetPath)) {
            targetPath = path.normalize(targetPath);
        } else {
            targetPath = path.normalize(path.join(process.targetPath(), targetPath));
        }
    }


    async.waterfall([
        function (cb) {
            fs.access(targetPath, cb);
        },
        function (cb) {
            fs.stat(targetPath, function (err, stats) {
                if (err) {
                    return cb(err);
                } else if (stats.isDirectory()) {
                    var pkgFile = path.join(targetPath, 'package', 'pkginfo.manifest');
                    fs.access(pkgFile, function (err) {
                        if (err) {
                            cb('Not support file type');
                        } else {
                            createSourceProject(targetPath, options, cb);
                        }
                    });
                } else {
                    var extName = path.extname(targetPath);
                    if (extName === '.gz' || extName === '.tar' || extName === '.bz2') {
                        createArchiveProject(targetPath, options, cb);
                    } else if (extName === '.zip') {
                        createBinaryProject(targetPath, options, cb);
                    } else {
                        cb('Not support file type');
                    }
                }
            });
        }
    ], callback);
};

function createSourceProject(targetPath, options, callback) {
    var pkgFile = path.join(targetPath, 'package', 'pkginfo.manifest');
    var hostOs = util.getTargetOS(os.platform(), os.arch());
    var targetOs = options.os || hostOs.os;

    util.package.getPkgListFromFile(pkgFile, function (err, packages) {
        if (err) {
            callback(err);
        } else {
            var pkgInfo = new PkgInfo(packages);
            callback(err, new TsSourceJob({
                name: path.basename(targetPath),
                os: targetOs,
                hostOs: hostOs,
                version: pkgInfo.version,
                srcPath: targetPath,
                packages: pkgInfo.getPackageNameList(),
                pkgInfo: pkgInfo,
                log: options.log
            }));
        }
    });
}

function createBinaryProject(targetPath, options, callback) {
    util.package.getPkgInfoFromPkgFile(targetPath, function (err, pkg) {
        if (err) {
            callback(err, null);
        } else {
            callback(null, new TsBinaryJob({
                name: pkg.name,
                os: pkg.os,
                version: pkg.version,
                filename: path.basename(targetPath),
                path: targetPath,
                pkgInfo: {packages: [pkg]}
            }));
        }
    });
}

function createArchiveProject(targetPath, options, callback) {
    var name = path.basename(targetPath);
    var job = new TsArchiveJob({
        name: name,
        filename: name,
        path: targetPath,
        log: options.log
    });

    callback(null, job);
}
