// //===-- TizenNativeAPIChecker.cpp -----------------------------------------*-
// C++ -*--//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
// Defines a checker to test memory leak in Tizen Native API's.
// Refer to TizenNativeAPIList.h to check the supported API's.
//
//===----------------------------------------------------------------------===//

#include "ClangSACheckers.h"
#include "clang/StaticAnalyzer/Core/Checker.h"
#include "clang/StaticAnalyzer/Core/PathSensitive/CallEvent.h"
#include "clang/StaticAnalyzer/Core/PathSensitive/CheckerContext.h"
#include "llvm/ADT/StringSwitch.h"
#include "clang/StaticAnalyzer/Core/BugReporter/BugType.h"
#include "TizenNativeAPIList.h"
#include <map>
#include <vector>

using namespace clang;
using namespace ento;
using namespace std;

namespace {

typedef SmallVector<SymbolRef, 2> SymbolVector;
typedef std::map<const std::string, std::pair<int, int>> AllocMap;
typedef std::map<const std::string, std::vector<int>> DeallocMap;

struct MemState {
private:
  // TODO: Handle escaped state. Currently we assume handles do not escape.
  // Uninit = -2, Deallocated = -1, Allocated >=0
  int state;
  MemState(int stateId) : state(stateId) {}

public:
  bool isAllocated() const { return (state != -1 && state != -2); }
  bool isDeallocated() const { return state == -1; }

  static MemState getAllocatedState(int stateId) { return MemState(stateId); }
  static MemState getDeallocatedState() { return MemState(-1); }

  int getStateID() const { return state; }
  bool operator==(const MemState &X) const { return state == X.state; }
  void Profile(llvm::FoldingSetNodeID &ID) const { ID.AddInteger(state); }
};

//

// ===========================Bug Visitor Calls===============================
// Bug visitor to track the flow which resulted in an error state.

class TizenNativeAPIBugVisitor
    : public BugReporterVisitorImpl<TizenNativeAPIBugVisitor> {

  // The allocated region tracked by the main analysis.
  SymbolRef Sym;
  bool IsLeak;

public:
  TizenNativeAPIBugVisitor(SymbolRef S, bool isLeak = false)
      : Sym(S), IsLeak(isLeak) {}

  virtual ~TizenNativeAPIBugVisitor() {}

  void Profile(llvm::FoldingSetNodeID &ID) const {
    static int X = 0;
    ID.AddPointer(&X);
    ID.AddPointer(Sym);
  }

  inline bool isAllocated(const MemState *S, const MemState *SPrev,
                          const Stmt *Stmt) {
    return (Stmt && isa<CallExpr>(Stmt) && (S && S->isAllocated()) &&
            (!SPrev || !SPrev->isAllocated()));
  }

  inline bool isDeallocated(const MemState *S, const MemState *SPrev,
                            const Stmt *Stmt) {
    return (Stmt && isa<CallExpr>(Stmt) && (S && S->isDeallocated()) &&
            (!SPrev || !SPrev->isDeallocated()));
  }

  PathDiagnosticPiece *VisitNode(const ExplodedNode *N,
                                 const ExplodedNode *PrevN,
                                 BugReporterContext &BRC, BugReport &BR);

  std::unique_ptr<PathDiagnosticPiece>
  getEndPath(BugReporterContext &BRC, const ExplodedNode *EndPathNode,
             BugReport &BR) {
    if (!IsLeak)
      return 0;

    PathDiagnosticLocation L = PathDiagnosticLocation::createEndOfPath(
        EndPathNode, *(EndPathNode->getSourceManager()));
    return llvm::make_unique<PathDiagnosticEventPiece>(
        L, BR.getDescription(), EndPathNode->getSourceManager(), false);
  }
};

// ========================TizenNativeAPIChecker==============================
// checks for improper merory usage of Tizen native API's
class TizenNativeAPIChecker
    : public Checker<check::PostCall, check::PreCall, check::DeadSymbols> {

  std::unique_ptr<BugType> UninitDeallocBugType;
  std::unique_ptr<BugType> DoubleDeallocBugType;
  std::unique_ptr<BugType> LeakBugType;
  // Map of allocator,pair<handleIndex,stateId>
  AllocMap mapAlloc;
  // Map of deallocator,vector of stateId(as multiple allocator
  // can have same deallocator)
  DeallocMap mapDealloc;

  vector<int> vecDeallocState;

  void reportUninitDealloc(const CallEvent &Call, CheckerContext &C) const;

  void reportDoubleDealloc(SymbolRef handle, const CallEvent &Call,
                           CheckerContext &C) const;

  void reportLeaks(SymbolVector &LeakedHandles, CheckerContext &C,
                   ExplodedNode *ErrNode) const;

public:
  TizenNativeAPIChecker();

  /// Process allocator calls.
  void checkPostCall(const CallEvent &Call, CheckerContext &C) const;
  /// Process deallocator calls.
  void checkPreCall(const CallEvent &Call, CheckerContext &C) const;

  void checkDeadSymbols(SymbolReaper &SymReaper, CheckerContext &C) const;
};

} // end anonymous namespace

/// The state of the checker is a map from tracked stream symbols to their
/// state. Let's store it in the ProgramState.
REGISTER_MAP_WITH_PROGRAMSTATE(RsrcMap, SymbolRef, MemState)

TizenNativeAPIChecker::TizenNativeAPIChecker() {

  for (int i = 0; i < int(sizeof(tizenMemAPIMap) / sizeof(TizenMemAPIMap));
       ++i) {
    string allocFn = tizenMemAPIMap[i].allocatorFn;
    int allocIndx = tizenMemAPIMap[i].allocIndx;
    string deallocFn = tizenMemAPIMap[i].deAllocatorFn;
    mapAlloc[allocFn] = make_pair(allocIndx, i);
    vecDeallocState.push_back(i);
    mapDealloc[deallocFn] = vecDeallocState;
  }

  UninitDeallocBugType.reset(
      new BugType("Unint Dealloc", "Tizen API Usage Error"));
  DoubleDeallocBugType.reset(
      new BugType("Double Dealloc", "Tizen API Usage Error"));
  LeakBugType.reset(
      new BugType("Tizen API Resource Leaks", "Tizen API Usage Error"));
}

void TizenNativeAPIChecker::checkPostCall(const CallEvent &Call,
                                          CheckerContext &C) const {
  //Disable Tizen Checker.
  return;
  const FunctionDecl *FD = dyn_cast_or_null<FunctionDecl>(Call.getDecl());
  if (!FD)
    return;
  string fnName = FD->getName().str();
  AllocMap::const_iterator I = mapAlloc.find(fnName);
  if (I == mapAlloc.end())
    return;

  ProgramStateRef State = C.getState();
  SValBuilder &svalBuilder = C.getSValBuilder();
  SymbolRef handle;
  SVal RetSVal = Call.getReturnValue();
  ASTContext &Ctx = C.getASTContext();
  QualType SizeTy = Ctx.getSizeType();
  DefinedSVal zeroVal = svalBuilder.makeIntVal(0, SizeTy);
  SVal Arg = Call.getArgSVal(I->second.first);
  Optional<Loc> CallArg = Arg.getAs<Loc>();
  if (!CallArg)
    return;

  // Assume that the API returns success else we would not have to track
  // the symbol generated by the function.
  SVal Success = svalBuilder.evalBinOp(State, BO_EQ, RetSVal, zeroVal, SizeTy);
  State = State->assume(Success.castAs<DefinedOrUnknownSVal>(), true);

  // Get the symbolic value corresponding to the handle.
  handle = Arg.getLocSymbolInBase();
  if (!handle) {
    const MemRegion *MR = Arg.getAsRegion();
    SVal V = State->getSVal(MR);
    handle = V.getAsLocSymbol();
  }

  QualType cmpTy = svalBuilder.getConditionType();
  SVal Result = svalBuilder.evalBinOpLL(State, BO_EQ, *CallArg,
                                        svalBuilder.makeNull(), cmpTy);
  // When the API is success we get a non NULL handle. Pass this info to state.
  State = State->assume(Result.castAs<DefinedOrUnknownSVal>(), false);

  // Generate the next transition (an edge in the exploded graph).
  State = State->set<RsrcMap>(handle,
                              MemState::getAllocatedState(I->second.second));

  C.addTransition(State);
}

void TizenNativeAPIChecker::checkPreCall(const CallEvent &Call,
                                         CheckerContext &C) const {
  //Disable Tizen Checker.
  return;
  bool foundMatchingDealloc = false;
  const FunctionDecl *FD = dyn_cast_or_null<FunctionDecl>(Call.getDecl());
  if (!FD)
    return;
  string fnName = FD->getName().str();
  DeallocMap::const_iterator I = mapDealloc.find(fnName);
  if (I == mapDealloc.end())
    return;

  ProgramStateRef State = C.getState();
  SymbolRef handle = 0;

  // Get the symbolic value corresponding to the handle.
  SymbolRef baseSym = Call.getArgSVal(0).getLocSymbolInBase();
  if (!baseSym)
    return;
  const SymbolDerived *SD = dyn_cast_or_null<SymbolDerived>(baseSym);
  if (!SD) {
    handle = baseSym;
  } else {
    const MemRegion *MR = SD->getRegion();
    if (!MR)
      return;
    handle = MR->getSymbolicBase()->getSymbol();
  }
  // Check if the handle has already been destroyed.
  const MemState *MS = State->get<RsrcMap>(handle);
  if (!MS)
    return;
  if (MS->isDeallocated()) {
    reportDoubleDealloc(handle, Call, C);
    return;
  }
  vector<int> vec = I->second;
  int stateID = MS->getStateID();
  for (vector<int>::iterator I = vec.begin(), E = vec.end(); I != E; ++I) {
    if ((*I) == stateID) {
      foundMatchingDealloc = true;
      break;
    }
  }
  if (foundMatchingDealloc) {
    // Matching Deallocator update state.
    State = State->set<RsrcMap>(handle, MemState::getDeallocatedState());
    C.addTransition(State);
  } else {
    // TODO: Mismatched delete call?
  }
}

static bool isLeaked(SymbolRef Sym, const MemState &MS, bool IsSymDead) {

  return (IsSymDead && MS.isAllocated());
}

void TizenNativeAPIChecker::checkDeadSymbols(SymbolReaper &SymReaper,
                                             CheckerContext &C) const {
  //Disable Tizen Checker.
  return;
  ProgramStateRef State = C.getState();
  SymbolVector LeakedHandles;
  RsrcMapTy TrackedStreams = State->get<RsrcMap>();
  for (RsrcMapTy::iterator I = TrackedStreams.begin(), E = TrackedStreams.end();
       I != E; ++I) {
    SymbolRef Sym = I->first;
    bool IsSymDead = SymReaper.isDead(Sym);

    // Collect leaked symbols.
    if (isLeaked(Sym, I->second, IsSymDead))
      LeakedHandles.push_back(Sym);

    // Remove the dead symbol from the streams map.
    if (IsSymDead)
      State = State->remove<RsrcMap>(Sym);
  }

  ExplodedNode *N = C.addTransition(State);
  reportLeaks(LeakedHandles, C, N);
}

void TizenNativeAPIChecker::reportDoubleDealloc(SymbolRef handle,
                                                const CallEvent &Call,
                                                CheckerContext &C) const {
  ProgramStateRef State = C.getState();
  ExplodedNode *ErrNode = C.addTransition(State);
  // If we've already reached this node on another path, return.
  if (!ErrNode)
    return;

  // Generate the report.
  BugReport *R =
      new BugReport(*DoubleDeallocBugType,
                    "Destroying a previously destroyed handle", ErrNode);
  R->addRange(Call.getSourceRange());
  R->markInteresting(handle);
  R->addVisitor(llvm::make_unique<TizenNativeAPIBugVisitor>(handle));
  C.emitReport(R);
}

void TizenNativeAPIChecker::reportUninitDealloc(const CallEvent &Call,
                                                CheckerContext &C) const {
  ProgramStateRef State = C.getState();
  ExplodedNode *ErrNode = C.addTransition(State);
  // If we've already reached this node on another path, return.
  if (!ErrNode)
    return;

  // Generate the report.
  BugReport *R =
      new BugReport(*UninitDeallocBugType,
                    "Destroying handle without allocating it.", ErrNode);
  R->addRange(Call.getSourceRange());
  C.emitReport(R);
}

void TizenNativeAPIChecker::reportLeaks(SymbolVector &LeakedHandles,
                                        CheckerContext &C,
                                        ExplodedNode *ErrNode) const {
  // Attach bug reports to the leak node.
  for (SmallVectorImpl<SymbolRef>::iterator I = LeakedHandles.begin(),
                                            E = LeakedHandles.end();
       I != E; ++I) {
    BugReport *R = new BugReport(*LeakBugType,
                                 "Handle allocated but not destroyed", ErrNode);
    R->markInteresting(*I);
    R->addVisitor(llvm::make_unique<TizenNativeAPIBugVisitor>(*I));
    C.emitReport(R);
  }
}

PathDiagnosticPiece *
TizenNativeAPIBugVisitor::VisitNode(const ExplodedNode *N,
                                    const ExplodedNode *PrevN,
                                    BugReporterContext &BRC, BugReport &BR) {

  ProgramStateRef state = N->getState();

  const MemState *RS = state->get<RsrcMap>(Sym);
  if (!RS)
    return 0;

  const Stmt *S = 0;
  const char *Msg = 0;
  StackHintGeneratorForSymbol *StackHint = 0;

  // Retrieve the associated statement.
  ProgramPoint ProgLoc = N->getLocation();
  if (Optional<StmtPoint> SP = ProgLoc.getAs<StmtPoint>())
    S = SP->getStmt();
  else if (Optional<CallExitEnd> Exit = ProgLoc.getAs<CallExitEnd>())
    S = Exit->getCalleeContext()->getCallSite();
  // If an assumption was made on a branch, it should be caught
  // here by looking at the state transition.
  else if (Optional<BlockEdge> Edge = ProgLoc.getAs<BlockEdge>()) {
    const CFGBlock *srcBlk = Edge->getSrc();
    S = srcBlk->getTerminator();
  }
  if (!S)
    return 0;

  ProgramStateRef statePrev = PrevN->getState();
  const MemState *RSPrev = statePrev->get<RsrcMap>(Sym);

  if (isAllocated(RS, RSPrev, S)) {
    Msg = "Handle is created";
    StackHint = new StackHintGeneratorForSymbol(Sym, "Returned created handle");
  } else if (isDeallocated(RS, RSPrev, S)) {
    Msg = "Handle is destroyed";
    StackHint =
        new StackHintGeneratorForSymbol(Sym, "Returning, handle was destroyed");
  }

  if (!Msg)
    return 0;
  assert(StackHint);

  // Generate the extra diagnostic.
  PathDiagnosticLocation Pos(S, *(N->getSourceManager()),
                             N->getLocationContext());
  return new PathDiagnosticEventPiece(Pos, Msg, N->getSourceManager(), true,
                                      StackHint);
}

// Finally register the checker.
void ento::registerTizenNativeAPIChecker(CheckerManager &mgr) {
  mgr.registerChecker<TizenNativeAPIChecker>();
}
