//===-- FileStreamChecker.cpp -----------------------------------------*- C++ -*--//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
// Defines a checker for proper use of fopen/fclose APIs.
//   - If a file has been closed with fclose, it should not be accessed again.
//   Accessing a closed file results in undefined behavior.
//   - If a file was opened with fopen, it must be closed with fclose before
//   the execution ends. Failing to do so results in a resource leak.
//
//===----------------------------------------------------------------------===//

#include "ClangSACheckers.h"
#include "clang/StaticAnalyzer/Core/Checker.h"
#include "clang/StaticAnalyzer/Core/BugReporter/BugType.h"
#include "clang/StaticAnalyzer/Core/PathSensitive/CallEvent.h"
#include "clang/StaticAnalyzer/Core/PathSensitive/CheckerContext.h"

using namespace clang;
using namespace ento;

namespace {
typedef llvm::SmallVector<const MemRegion*,2> RegionVector;
typedef std::pair<const Stmt*, const MemRegion*> LeakInfo;

struct StreamState {
private:
  enum Kind { Opened, Closed ,Escaped} K;
  StreamState(Kind InK) : K(InK) { }

public:
  bool isOpened() const { return K == Opened; }
  bool isClosed() const { return K == Closed; }
  bool isEscaped() const { return K == Escaped; }

  static StreamState getOpened() { return StreamState(Opened); }
  static StreamState getClosed() { return StreamState(Closed); }
  static StreamState getEscaped() { return StreamState(Escaped); }

  bool operator==(const StreamState &X) const {
    return K == X.K;
  }
  void Profile(llvm::FoldingSetNodeID &ID) const {
    ID.AddInteger(K);
  }
};

class FileStreamBugVisitor : public BugReporterVisitorImpl<FileStreamBugVisitor> {

  // The allocated region tracked by the main analysis.
  const MemRegion* MR;
  bool IsLeak;

  public:
    FileStreamBugVisitor(const MemRegion* mr, bool isLeak = false)
       : MR(mr), IsLeak(isLeak) {}

    virtual ~FileStreamBugVisitor() {}

    void Profile(llvm::FoldingSetNodeID &ID) const {
      static int X = 0;
      ID.AddPointer(&X);
      ID.AddPointer(MR);
    }

    inline bool isOpened(const StreamState *S, const StreamState *SPrev,
                            const Stmt *Stmt) {
      return (Stmt && isa<CallExpr>(Stmt) &&
              (S && S->isOpened()) && (!SPrev || !SPrev->isOpened()));
    }

    inline bool isClosed(const StreamState *S, const StreamState *SPrev,
                           const Stmt *Stmt) {
      return (Stmt && isa<CallExpr>(Stmt) &&
              (S && S->isClosed()) && (!SPrev || !SPrev->isClosed()));
    }
    
    PathDiagnosticPiece* VisitNode(const ExplodedNode *N,
                                   const ExplodedNode *PrevN,
                                   BugReporterContext &BRC,
                                   BugReport &BR);

    std::unique_ptr<PathDiagnosticPiece> getEndPath(BugReporterContext &BRC,
                                    const ExplodedNode *EndPathNode,
                                    BugReport &BR) {
      if (!IsLeak)
        return 0;

      PathDiagnosticLocation L =
        PathDiagnosticLocation::createEndOfPath(EndPathNode,
                                                *(EndPathNode->getSourceManager()));
      return llvm::make_unique<PathDiagnosticEventPiece>(L, BR.getDescription(), EndPathNode->getSourceManager(), false);
    }

};

class FileStreamChecker
    : public Checker<check::PostCall, check::PreCall, check::DeadSymbols,
                     check::Bind, check::RegionChanges> {

  mutable IdentifierInfo *IIfopen, *IIfreopen, *IIfclose;

  mutable std::unique_ptr<BugType> DoubleCloseBugType;
  mutable std::unique_ptr<BugType> LeakBugType;

  void initIdentifierInfo(ASTContext &Ctx) const;

  void reportDoubleClose(const MemRegion* FileDescSym,
                         const CallEvent &Call,
                         CheckerContext &C) const;

  void reportLeaks(RegionVector LeakedStreams,
                   CheckerContext &C,
                   ExplodedNode *ErrNode) const;
  
public:
  FileStreamChecker();
  
  std::unique_ptr<FileStreamBugVisitor> 
  getFileStreamBugVisitor(const MemRegion* mr, bool isLeak = false) const {
    return llvm::make_unique<FileStreamBugVisitor>(mr,isLeak);
  }
  /// Process fopen.
  void checkPostCall(const CallEvent &Call, CheckerContext &C) const;
  /// Process fclose.
  void checkPreCall(const CallEvent &Call, CheckerContext &C) const;

  void checkDeadSymbols(SymbolReaper &SymReaper, CheckerContext &C) const;

  void checkBind(SVal location, SVal val, const Stmt *S,
                 CheckerContext &C) const;

  bool guaranteedNotToCloseFile(const CallEvent &Call) const;

  ProgramStateRef checkRegionChanges(
      ProgramStateRef state, const InvalidatedSymbols *invalidated,
      ArrayRef<const MemRegion *> ExplicitRegions,
      ArrayRef<const MemRegion *> Regions, const CallEvent *Call) const;
  bool wantsRegionChangeUpdate(ProgramStateRef state) const { return true; }
};

} // end anonymous namespace

/// The state of the checker is a map from tracked stream symbols to their
/// state. Let's store it in the ProgramState.
REGISTER_MAP_WITH_PROGRAMSTATE(StreamMap, const MemRegion*, StreamState)
REGISTER_MAP_WITH_PROGRAMSTATE(StreamValMap, const MemRegion*, SVal)


PathDiagnosticPiece *
FileStreamBugVisitor::VisitNode(const ExplodedNode *N,
                                const ExplodedNode *PrevN,
                                BugReporterContext &BRC,
                                BugReport &BR) {

  ProgramStateRef state = N->getState();
  ProgramStateRef statePrev = PrevN->getState();

  const StreamState *RS = state->get<StreamMap>(MR);
  const StreamState *RSPrev = statePrev->get<StreamMap>(MR);
  if (!RS)
    return 0;

  const Stmt *S = 0;
  const char *Msg = 0;
  StackHintGeneratorForSymbol *StackHint = 0;

  // Retrieve the associated statement.
  ProgramPoint ProgLoc = N->getLocation();
  if (Optional<StmtPoint> SP = ProgLoc.getAs<StmtPoint>())
    S = SP->getStmt();
  else if (Optional<CallExitEnd> Exit = ProgLoc.getAs<CallExitEnd>())
    S = Exit->getCalleeContext()->getCallSite();
  // If an assumption was made on a branch, it should be caught
  // here by looking at the state transition.
  else if (Optional<BlockEdge> Edge = ProgLoc.getAs<BlockEdge>()) {
    const CFGBlock *srcBlk = Edge->getSrc();
    S = srcBlk->getTerminator();
  }
  if (!S)
    return 0;

  if (isOpened(RS, RSPrev, S)) {
    if (const SymbolicRegion *SymMR =
        dyn_cast<SymbolicRegion>(MR)) {
      SymbolRef Sym = SymMR->getSymbol();
      Msg = "File is opened";
      StackHint = new StackHintGeneratorForSymbol(Sym,
                                                  "Opened File");
    }
  }
  else if (isClosed(RS, RSPrev, S)) {
    if (const SymbolicRegion *SymMR =
        dyn_cast<SymbolicRegion>(MR)) {
      SymbolRef Sym = SymMR->getSymbol();
      Msg = "File is closed";
      StackHint = new StackHintGeneratorForSymbol(Sym,
                                                  "Closed file");
    }
  }

  if (!Msg)
    return 0;
  assert(StackHint);

  // Generate the extra diagnostic.
  PathDiagnosticLocation Pos(S, *(N->getSourceManager()),
                             N->getLocationContext());
  return new PathDiagnosticEventPiece(Pos, Msg, N->getSourceManager(), true,
                                      StackHint);
}

FileStreamChecker::FileStreamChecker() : IIfopen(0), IIfreopen(0), IIfclose(0) {
  // Initialize the bug types.
  DoubleCloseBugType.reset(new BugType(this,"Double fclose",
                                       "Unix Stream API Error"));

  LeakBugType.reset(new BugType(this,"Resource Leak",
                                "Unix Stream API Error"));
  // Sinks are higher importance bugs as well as calls to assert() or exit(0).
  LeakBugType->setSuppressOnSink(true);
}

void FileStreamChecker::checkPostCall(const CallEvent &Call,
                                        CheckerContext &C) const {
  initIdentifierInfo(C.getASTContext());
  RegionVector LeakedStreams;
  if (!Call.isGlobalCFunction())
    return;

  const FunctionDecl *FD = dyn_cast_or_null<FunctionDecl>(Call.getDecl());
  if (!FD)
    return ;



  if (Call.getCalleeIdentifier() != IIfopen && FD->getNameAsString() != "fopen")
    return;

  // Get the symbolic value corresponding to the file handle.
  const MemRegion* FileDesc = Call.getReturnValue().getAsRegion();
  if (!FileDesc)
    return;
  ProgramStateRef State = C.getState();
  const StreamState *SS = State->get<StreamMap>(FileDesc);
  if (SS && SS->isOpened()) {
    LeakedStreams.push_back(FileDesc);
    ExplodedNode *N = C.addTransition(State);
    reportLeaks(LeakedStreams, C, N);
  }
  // Generate the next transition (an edge in the exploded graph).
  State = State->set<StreamMap>(FileDesc, StreamState::getOpened());
  C.addTransition(State);
}

void FileStreamChecker::checkPreCall(const CallEvent &Call,
                                       CheckerContext &C) const {
  initIdentifierInfo(C.getASTContext());

  if (!Call.isGlobalCFunction())
    return;

  const FunctionDecl *FD = dyn_cast_or_null<FunctionDecl>(Call.getDecl());
  if (!FD)
    return ;

  if (Call.getCalleeIdentifier() != IIfclose && FD->getNameAsString() != "fclose"
      && Call.getCalleeIdentifier() != IIfreopen && FD->getNameAsString() != "freopen")
    return;

  if (Call.getNumArgs() < 1)
    return;
  
  if (Call.getCalleeIdentifier() == IIfreopen || FD->getNameAsString() == "freopen") {
   //TODO: Mark the stream as escaped for now. We need to create a dup of stream and track the same.
   // Currently not supported in Bada Potential check. Implement in second phase.
    if (Call.getNumArgs() != 3)
      return;
    const MemRegion* FileDesc = Call.getArgSVal(2).getAsRegion();
    if (!FileDesc)
      return;
    ProgramStateRef State = C.getState();
    State = State->set<StreamMap>(FileDesc, StreamState::getEscaped());
    C.addTransition(State);
    return;
  }

  // Get the symbolic value corresponding to the file handle.
  const MemRegion* FileDesc = Call.getArgSVal(0).getAsRegion();
  if (!FileDesc)
    return;

  // Check if the stream has already been closed.
  ProgramStateRef State = C.getState();
  const StreamState *SS = State->get<StreamMap>(FileDesc);
  if (SS && SS->isClosed()) {
    reportDoubleClose(FileDesc, Call, C);
    return;
  }

  // Generate the next transition, in which the stream is closed.
  State = State->set<StreamMap>(FileDesc, StreamState::getClosed());
  C.addTransition(State);
}

void FileStreamChecker::checkBind(SVal location, SVal val, const Stmt*S,
                                  CheckerContext &C) const {
  if (const MemRegion *FileDesc = val.getAsRegion()) {
    ProgramStateRef State = C.getState();
    const StreamState *SS = State->get<StreamMap>(FileDesc);
    if (!SS)
      return;
    if (!SS->isOpened())
      return;
    State = State->set<StreamValMap>(FileDesc, location);
    C.addTransition(State);
    
  }
}

static bool isLeaked(const MemRegion *Sym, const StreamState &SS,
                     bool IsSymDead, ProgramStateRef State) {
  if (IsSymDead && SS.isOpened()) {
    ConstraintManager &CMgr = State->getConstraintManager();
    ConditionTruthVal OpenFailed = CMgr.isNull(State, Sym);
    return !OpenFailed.isConstrainedTrue();
  }
  return false;
}

void FileStreamChecker::checkDeadSymbols(SymbolReaper &SymReaper,
                                           CheckerContext &C) const {
  ProgramStateRef State = C.getState();
  RegionVector LeakedStreams;
  StreamMapTy TrackedStreams = State->get<StreamMap>();
  for (StreamMapTy::iterator I = TrackedStreams.begin(),
                             E = TrackedStreams.end(); I != E; ++I) {
    const MemRegion* Sym = I->first;
    bool IsSymLive = SymReaper.isLiveRegion(Sym);

    // Collect leaked symbols.
    if (isLeaked(Sym, I->second, !IsSymLive, State))
      LeakedStreams.push_back(Sym);

    // Remove the dead symbol from the streams map.
    if (!IsSymLive)
      State = State->remove<StreamMap>(Sym);
  }

  ExplodedNode* N = C.addTransition(State);
  reportLeaks(LeakedStreams, C, N);
}

void FileStreamChecker::reportDoubleClose(const MemRegion* FileDescSym,
                                            const CallEvent &Call,
                                            CheckerContext &C) const {
  ProgramStateRef State = C.getState();
  ExplodedNode *ErrNode = C.addTransition(State);
  // If we've already reached this node on another path, return.
  if (!ErrNode)
    return;

  // Generate the report.
  BugReport *R = new BugReport(*DoubleCloseBugType,
      "Closing a previously closed file stream", ErrNode);
  R->addRange(Call.getSourceRange());
  R->addVisitor(getFileStreamBugVisitor(FileDescSym));
  R->markInteresting(FileDescSym);
  C.emitReport(R);
}

void FileStreamChecker::reportLeaks(RegionVector LeakedStreams,
                                               CheckerContext &C,
                                               ExplodedNode *ErrNode) const {
  ProgramStateRef State = C.getState();
  for (llvm::SmallVector<const MemRegion*, 2>::iterator
      I = LeakedStreams.begin(), E = LeakedStreams.end(); I != E; ++I) {
    SmallString<200> buf;
    llvm::raw_svector_ostream os(buf);
    const MemRegion* MemReg = *I;
    if (!MemReg)
      continue;
    const SVal* V = State->get<StreamValMap>(MemReg);
    if (!V)
      continue;
    const MemRegion* boundVarReg = V->getAsRegion();
    if (!boundVarReg)
      return;
    const VarRegion *VR = dyn_cast<VarRegion>(boundVarReg);
    const VarDecl *VD;
    if (VR)
      VD = VR->getDecl();
    else 
      VD = NULL;
    os << "Opened file is never closed; potential resource leak";
    if (VD) {
      os << " of resource pointed to by '";
      os << VD->getName();
      os << '\'';
    }

    BugReport *R = new BugReport(*LeakBugType,
        os.str(), ErrNode);
    R->markInteresting(*I);
    R->addVisitor(getFileStreamBugVisitor(MemReg, true));
    C.emitReport(R);
  }
}

void FileStreamChecker::initIdentifierInfo(ASTContext &Ctx) const {
  if (IIfopen)
    return;
  IIfopen = &Ctx.Idents.get("fopen");
  IIfreopen = &Ctx.Idents.get("freopen");
  IIfclose = &Ctx.Idents.get("fclose");
}

void ento::registerFileStreamChecker(CheckerManager &mgr) {
  mgr.registerChecker<FileStreamChecker>();
}

bool FileStreamChecker::guaranteedNotToCloseFile(const CallEvent &Call) const {

  // If it's not in a system header, assume it might close a file.
  if (!Call.isInSystemHeader())
    return false;

  // Handle cases where we know a buffer's /address/ can escape.
  if (Call.argumentsMayEscape())
    return false;

  // Note, even though fclose closes the file, we do not list it here
  // since the checker is modeling the call.

  return true;
}

ProgramStateRef FileStreamChecker::checkRegionChanges(
    ProgramStateRef state, const InvalidatedSymbols *invalidated,
    ArrayRef<const MemRegion *> ExplicitRegions,
    ArrayRef<const MemRegion *> Regions, const CallEvent *Call) const {

  if (!Call)
    return state;
  // If we know that the call cannot close a file, there is nothing to do.
  if (guaranteedNotToCloseFile(*Call)) {
    return state;
  }
  if (!invalidated)
    return state;
  for (ArrayRef<const MemRegion *>::iterator I = ExplicitRegions.begin(),
                                             E = ExplicitRegions.end();
       I != E; ++I) {
    state = state->remove<StreamMap>(*I);
  }
  return state;
}
