//===--- XMLDiagnostics.cpp - XML Diagnostics for Paths ----*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
//  This file defines the XMLDiagnostics object.
//
//===----------------------------------------------------------------------===//

#include "clang/StaticAnalyzer/Core/PathDiagnosticConsumers.h"
#include "clang/AST/ASTContext.h"
#include "clang/AST/Decl.h"
#include "clang/Basic/FileManager.h"
#include "clang/Basic/SourceManager.h"
#include "clang/Lex/Lexer.h"
#include "clang/Lex/Preprocessor.h"
#include "clang/Rewrite/Core/HTMLRewrite.h"
#include "clang/Rewrite/Core/Rewriter.h"
#include "clang/StaticAnalyzer/Core/BugReporter/PathDiagnostic.h"
#include "llvm/Support/FileSystem.h"
#include "llvm/Support/MemoryBuffer.h"
#include "llvm/Support/Path.h"
#include "llvm/Support/raw_ostream.h"
#include "llvm/Support/Debug.h"

using namespace clang;
using namespace ento;

namespace {

class XMLDiagnostics : public PathDiagnosticConsumer {
  StringRef ReportFN;
  const Preprocessor &PP;
  int tabs;

public:
  XMLDiagnostics(const std::string& prefix, const Preprocessor &pp) : ReportFN(prefix), PP(pp), tabs(0) {}
  virtual ~XMLDiagnostics() { FlushDiagnostics(NULL); }

  virtual StringRef getName() const { return "XMLDiagnostics"; }
  bool supportsLogicalOpControlFlow() const override { return true; }
  bool supportsCrossFileDiagnostics() const override { return true; }

  llvm::raw_ostream &indent(llvm::raw_ostream &os) {
    for(int i = 0; i < tabs; ++i) os << "   "; return os;
  }

  void emitReportHeader(llvm::raw_ostream &os) {
    os << "<?xml version= \"1.0\"?>\n";
  }

  void emitStartField(llvm::raw_ostream &os, StringRef fName) {
    indent(os) << '<' << fName << ">\n"; ++tabs;
  }

  void emitEndField(llvm::raw_ostream &os, StringRef fName) {
    --tabs; indent(os) << "</" << fName << ">\n";
  }

  template<typename T>
  void emitField(llvm::raw_ostream &os, StringRef fName, T val) {
      indent(os) << '<' << fName << '>' << val << "</" << fName << ">\n";
  }

  void emitDescription(llvm::raw_ostream &os, const char* val) {
    indent(os) << "<DES><![CDATA[" << val << "]]></DES>\n";
  }

  void emitLocation(llvm::raw_ostream &os, const PathDiagnosticPiece& P) {
    FullSourceLoc Pos = P.getLocation().asLocation();
    if (!Pos.isValid()) return;

    FileID FID = Pos.getExpansionLoc().getFileID();
    if(const FileEntry *Entry = Pos.getManager().getFileEntryForID(FID)) {
      SmallString<200> filePath = StringRef(Entry->getName());
      llvm::sys::fs::make_absolute(filePath);
      emitField(os, "FILE", filePath.c_str());
      emitField(os, "LINE", Pos.getExpansionLineNumber());
      emitField(os, "COLUMN", Pos.getExpansionColumnNumber());
    }
  }

  void emitEvents(llvm::raw_ostream &os, const PathDiagnosticPiece& P, unsigned &id, StringRef desPrefix = "") {
    std::string Des(desPrefix);
    Des = Des + P.getString().data();
    if (Des.empty()) {
      if (auto MP = dyn_cast<PathDiagnosticMacroPiece>(&P)) {
        if (MP->subPieces.begin() != MP->subPieces.end()) {
          for (auto &EP : MP->subPieces)
            emitEvents(os, *EP, ++id, "Within macro expansion, ");
          return;
        }
        Des = "Within the expansion of the macro";
      } else {
        Des = "Within expression";
      }
    }

    emitStartField(os, "Event");
    emitField(os, "Id", id);
    emitLocation(os, P);
    emitDescription(os, Des.data());
    emitEndField(os, "Event");

  }

  bool emitReport(llvm::raw_ostream &os, const PathDiagnostic &D, unsigned bugNum) {
    PathPieces path = D.path.flatten(false);
    emitField(os, "ReportId", bugNum);
    emitStartField(os, "Report");
    emitLocation(os, *(*path.rbegin()));
    emitField(os, "Type", D.getBugType().data());
    emitField(os, "Severity", D.getSeverity().data());
    emitDescription(os, D.getVerboseDescription().data());

    if(path.begin() != path.end()) {
      unsigned n = 0;
      emitStartField(os, "Events");
      for(const auto &it : path) emitEvents(os, *it, ++n);
      emitEndField(os, "Events");
    }
    emitEndField(os, "Report");
    return true;
  }

  void emitReports(llvm::raw_ostream &os, std::vector<const PathDiagnostic *> &Diags) {
    emitReportHeader(os);
    emitStartField(os, "ConsolidatedReport");
    int bugCnt = 1; tabs = 0;
    for(const auto &it : Diags) {
      if(emitReport(os, *it, bugCnt))
        bugCnt++;
    }
    emitEndField(os, "ConsolidatedReport");
  }

  virtual void FlushDiagnosticsImpl(std::vector<const PathDiagnostic *> &Diags, FilesMade *filesMade) {
//    emitReports(llvm::dbgs(), Diags);
    int FD;
    if (!llvm::sys::fs::is_directory(ReportFN)) {
      if (auto EC = openFileForWrite(ReportFN, FD, llvm::sys::fs::F_None)) {
        llvm::errs() << "warning: could not create xml " << ReportFN << " file\n";
        return;
      }
    } else {
      SmallString<128> Model, ResultFile;
      llvm::sys::path::append(Model, ReportFN, "report-%%%%%%.xml");
      if (auto EC = llvm::sys::fs::createUniqueFile(Model.str(), FD, ResultFile)) {
        llvm::errs() << "warning: could not create xml " << ResultFile << " file\n";
        return;
      }
    }

    std::string s;
    llvm::raw_string_ostream os(s);
    emitReports(os, Diags);

    llvm::raw_fd_ostream(FD, true) << os.str();
  }

};

} // end anonymous namespace

void ento::createXMLDiagnosticConsumer(AnalyzerOptions &AnalyzerOpts,
                                        PathDiagnosticConsumers &C,
                                        const std::string& prefix,
                                        const Preprocessor &PP) {
  C.push_back(new XMLDiagnostics(prefix, PP));
}







