//===--- tools/APIChecker/APICheckerConsumer.cpp --------------===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
//  This file implements the ASTConsumer for APIChecker Tool
//
//===----------------------------------------------------------------------===//

#include "APICheckerConsumer.h"
#include "llvm/Support/raw_ostream.h"
#include "clang/AST/CommentVisitor.h"
#include "clang/Tooling/CommonOptionsParser.h"
#include <sstream>
#include <algorithm>

using namespace clang;
using namespace clang::tooling;
using namespace llvm;

extern cl::OptionCategory APICheckerOptions;
extern cl::opt<std::string> OutputFilename;
extern cl::opt<std::string> PrivilegeListOpt;

static cl::opt<std::string>
    APIVersionOpt("api-version",
                  cl::desc("Current API Version in manifest File."),
                  cl::cat(APICheckerOptions));

static cl::opt<std::string> ProfileNameOpt("current-profile",
                                           cl::desc("Current device profile."),
                                           cl::cat(APICheckerOptions));

static std::vector<std::string>
getDefinedPrivilegeList(const BlockCommandComment *BCC) {
  std::vector<std::string> Result;
  if (ParagraphComment *PComment =
          dyn_cast<ParagraphComment>(*(BCC->child_begin()))) {
    for (auto PI = PComment->child_begin(), PE = PComment->child_end();
         PI != PE; ++PI) {
      TextComment *textComment = dyn_cast<TextComment>(*PI);
      if (!textComment)
        continue;
      std::string privilege = textComment->getText().str();
      privilege.erase(
          std::remove_if(privilege.begin(), privilege.end(), [](char c) {
            return (c == '\r' || c == '\t' || c == ' ' || c == '\n' ||
                    c == '%');
          }), privilege.end());
      Result.push_back(privilege);
    }
  }
  return Result;
}

static void getVersionInt(const std::string &VerString, int &Major, int &Minor,
                          int &SubMinor, int &SubSubMinor) {
  std::istringstream is(VerString);
  int counter = 1;
  std::string part;
  while (getline(is, part, '.')) {
    int Version = std::stoi(part);
    if (counter == 1)
      Major = Version;
    else if (counter == 2)
      Minor = Version;
    else if (counter == 3)
      SubMinor = Version;
    else if (counter == 4)
      SubSubMinor = Version;
    ++counter;
  }
}

static bool compareVersion(const std::string &Version,
                           const std::string &CurrentVersion) {
  int Major = 0;
  int Minor = 0;
  int SubMinor = 0;
  int SubSubMinor = 0;
  int CMajor = 0;
  int CMinor = 0;
  int CSubMinor = 0;
  int CSubSubMinor = 0;
  getVersionInt(Version, Major, Minor, SubMinor, SubSubMinor);
  getVersionInt(CurrentVersion, CMajor, CMinor, CSubMinor, CSubSubMinor);

  if (Major > CMajor)
    return false;
  if ((Major == CMajor) && (Minor > CMinor))
    return false;
  if ((Major == CMajor) && (Minor == CMinor) && (SubMinor > CSubMinor))
    return false;
  if ((Major == CMajor) && (Minor == CMinor) && (SubMinor == CSubMinor) &&
      SubSubMinor > CSubSubMinor)
    return false;
  return true;
}

static bool isVersion(std::string Word) {
  size_t pos = Word.find('.');
  if (pos != std::string::npos)
    return true;
  return false;
}

static std::string changeToUpper(std::string ProfileName) {
  std::transform(ProfileName.begin(), ProfileName.end(), ProfileName.begin(),
                 ::toupper);
  return ProfileName;
}

static std::string getTizenVersion(std::string TizenVersion) {
  std::stringstream Stream(TizenVersion);
  std::string Word;
  std::string SupportedApiVersion;
  Stream >> Word;
  while (!Word.empty()) {
    if (isVersion(Word)) {
      SupportedApiVersion = Word;
      break;
    }
    Stream >> Word;
  }
  return SupportedApiVersion;
}

static bool findStringIC(const std::string &FullString,
                         const std::string &SearchString) {
  auto it =
      std::search(FullString.begin(), FullString.end(), SearchString.begin(),
                  SearchString.end(), [](char ch1, char ch2) {
                    return ::toupper(ch1) == ::toupper(ch2);
                  });
  return (it != FullString.end());
}

static bool compareString(std::string str1, std::string str2) {
  str1.erase(remove_if(str1.begin(), str1.end(), isspace), str1.end());
  str2.erase(remove_if(str2.begin(), str2.end(), isspace), str2.end());
  return (str1 == str2);
}

// APICheckerConsumer HandleTopLevelDecl :
// Visit All Top Level Function Declerations and Call Recursive Visitor to
// traverse all statements in the Function.
bool APICheckerConsumer::HandleTopLevelDecl(DeclGroupRef DR) {
  for (auto D : DR)
    Visitor.TraverseDecl(D);
  return true;
}

class CommentChecker {
private:
  FullComment *DoxyComment;
  std::string ApiVersion;
  std::string DetectedApiVersion;
  std::string Profile;
  bool IsDeprecatedFn;
  bool IsValidAPIUsage;
  std::unordered_map<std::string, bool> PrivilegeMap;
  std::vector<std::string> DefinedPrivilegeList;
  std::vector<std::string> UnMatchedPrivilege;

  bool checkIfDeprecated(const BlockCommandComment *BCC);
  bool checkAPIUsage(const BlockCommandComment *BCC, Comment *PrevComment);
  bool getPrevBlockText(Comment *PrevComment, std::string &ProfileName);
  void populateUnMatchedPrivilege(std::vector<std::string> PrivilegeList);
  void updateUsedPrivileges(std::vector<std::string> PrivilegeList);

public:
  CommentChecker(FullComment *Comment);
  bool CheckFunction();
  void ReportProblems(const CallExpr *CE, DiagnosticsEngine *Diag);
  void ReportEnumProblems(const VarDecl *VD, StringRef TypeDefName,
                          DiagnosticsEngine *Diag);
};

CommentChecker::CommentChecker(FullComment *Comment) : DoxyComment(Comment) {
  ApiVersion = APIVersionOpt;
  Profile = changeToUpper(ProfileNameOpt);
  IsDeprecatedFn = false;
  IsValidAPIUsage = true;
  std::string PrivilegeList(PrivilegeListOpt.c_str());
  char *PrivString = const_cast<char *>(PrivilegeList.c_str());
  char *pch = strtok(PrivString, " ,");
  while (pch != NULL) {
    PrivilegeMap[std::string(pch)] = true;
    pch = strtok(NULL, " ,");
  }
}

bool CommentChecker::checkIfDeprecated(const BlockCommandComment *BCC) {

  ParagraphComment *PComment =
      dyn_cast<ParagraphComment>(*(BCC->child_begin()));
  std::string Text;
  if (!PComment)
    return false;
  for (auto PI = PComment->child_begin(), PE = PComment->child_end(); PI != PE;
       ++PI) {
    TextComment *textComment = dyn_cast<TextComment>(*PI);
    if (!textComment)
      continue;
    Text = textComment->getText().str();
    break;
  }
  size_t Pos = Text.find("Deprecated since");
  // Check if deprecated for all versions
  if (Pos == std::string::npos)
    return true;
  std::stringstream Stream(Text);
  std::string Word;
  Stream >> Word;
  while (!Word.empty()) {
    if (isVersion(Word)) {
      DetectedApiVersion = Word;
      break;
    }
    Stream >> Word;
  }
  if (DetectedApiVersion.empty())
    return false;
  return compareVersion(DetectedApiVersion, ApiVersion);
}

bool CommentChecker::getPrevBlockText(Comment *PrevComment,
                                      std::string &ProfileName) {
  if (!PrevComment)
    return false;
  ParagraphComment *PComment = dyn_cast<ParagraphComment>(PrevComment);
  if (!PComment) {
    BlockCommandComment *BComment = dyn_cast<BlockCommandComment>(PrevComment);
    if (!BComment)
      return false;
    PComment = dyn_cast<ParagraphComment>(*(BComment->child_begin()));
    if (!PComment)
      return false;
  }
  auto I = PComment->child_end();
  --I;
  TextComment *Text = dyn_cast<TextComment>(*I);
  if (!Text)
    return false;
  std::string InfoText = Text->getText().str();
  ProfileName = InfoText;
  return true;
}

bool CommentChecker::checkAPIUsage(const BlockCommandComment *BCC,
                                   Comment *PrevComment) {

  ParagraphComment *PComment =
      dyn_cast<ParagraphComment>(*(BCC->child_begin()));
  std::string Text;
  std::string ProfileName;
  std::string SupportedVersion = "";
  if (!PComment)
    return true;
  bool ShouldCheckInPreviousBlock = getPrevBlockText(PrevComment, ProfileName);

  // Check for @since_tizen X.Y
  auto CI = PComment->child_end();
  auto PI = --CI;
  TextComment *TextInfo = dyn_cast<TextComment>(*PI);
  if (TextInfo) {
    ProfileName.erase(
        std::remove_if(ProfileName.begin(), ProfileName.end(), [](char c) {
          return (c == '\r' || c == '\t' || c == ' ');
        }), ProfileName.end());
    if (ProfileName.empty() &&
        (PComment->child_begin() != PComment->child_end())) {
      TextComment *textComment =
          dyn_cast<TextComment>(*PComment->child_begin());
      if (!textComment)
        return true;
      Text = textComment->getText().str();
      if (Text.find('.') != std::string::npos) {
        SupportedVersion = getTizenVersion(Text);
        return compareVersion(SupportedVersion, ApiVersion);
      }
    }
  }

  // Check for other types of @since
  for (auto PI = PComment->child_begin(), PE = PComment->child_end(); PI != PE;
       ++PI) {
    TextComment *textComment = dyn_cast<TextComment>(*PI);
    if (!textComment)
      continue;
    Text = textComment->getText().str();
    if (findStringIC(Text, Profile) && Text.find('.') != std::string::npos) {
      SupportedVersion = getTizenVersion(Text);
      return compareVersion(SupportedVersion, ApiVersion);
    } else if (ShouldCheckInPreviousBlock &&
               compareString(ProfileName, Profile) &&
               Text.find('.') != std::string::npos) {
      SupportedVersion = getTizenVersion(Text);
      return compareVersion(SupportedVersion, ApiVersion);
    }
  }
  return true;
}

void CommentChecker::populateUnMatchedPrivilege(
    std::vector<std::string> PrivilegeList) {

  for (auto I = PrivilegeList.begin(), E = PrivilegeList.end(); I != E; ++I) {
    std::string Privi = *I;
    if (!PrivilegeMap[Privi] && !Privi.empty()) {
      UnMatchedPrivilege.push_back(Privi);
    }
  }
}

void CommentChecker::updateUsedPrivileges(
    std::vector<std::string> PrivilegeList) {
  bool WriteToFile = !OutputFilename.empty();
  if (!WriteToFile)
    return;
  std::ofstream outfile;
  outfile.open(OutputFilename.c_str(), std::ios_base::app);
  for (auto I = PrivilegeList.begin(), E = PrivilegeList.end(); I != E; ++I)
    outfile << *I << "\n";
  outfile.close();
}

bool CommentChecker::CheckFunction() {
  Comment *PrevComment = NULL;
  for (auto I = DoxyComment->child_begin(), E = DoxyComment->child_end();
       I != E; ++I) {
    const Comment *Child = *I;
    if (!Child)
      continue;

    if (Child->getCommentKind() == Comment::BlockCommandCommentKind) {
      const BlockCommandComment *BCC = cast<BlockCommandComment>(Child);
      unsigned CommandID = BCC->getCommandID();
      if (CommandID == CommandTraits::KCI_deprecated) {
        IsDeprecatedFn = checkIfDeprecated(BCC);
        if (IsDeprecatedFn)
          return true;
      } else if (CommandID == CommandTraits::KCI_privilege) {
        DefinedPrivilegeList = getDefinedPrivilegeList(BCC);
      } else if (CommandID == CommandTraits::KCI_since_tizen) {
        IsValidAPIUsage = checkAPIUsage(BCC, PrevComment);
        if (!IsValidAPIUsage)
          return true;
      }
    }
    PrevComment = *I;
  }
  populateUnMatchedPrivilege(DefinedPrivilegeList);
  updateUsedPrivileges(DefinedPrivilegeList);
  return true;
}

void CommentChecker::ReportProblems(const CallExpr *CE,
                                    DiagnosticsEngine *Diag) {
  const FunctionDecl *FD = CE->getDirectCallee();
  // Check if API is Deprecated.
  if (IsDeprecatedFn) {
    unsigned Warning = Diag->getCustomDiagID(DiagnosticsEngine::Warning,
                                             "Function %0 is deprecated in %1");
    Diag->Report(CE->getLocStart(), Warning) << FD->getNameAsString()
                                             << ApiVersion.c_str();
    return;
  }

  // Check if API usage is valid.
  if (!IsValidAPIUsage) {
    unsigned Warning = Diag->getCustomDiagID(
        DiagnosticsEngine::Warning, "Function %0 is not supported in %1");
    Diag->Report(CE->getLocStart(), Warning) << FD->getNameAsString()
                                             << ApiVersion.c_str();
    return;
  }

  // Check if some privileges are missing.
  if (!UnMatchedPrivilege.empty()) {
    std::string RequiredPrivilege = "";
    for (auto I = UnMatchedPrivilege.begin(), E = UnMatchedPrivilege.end();
         I != E;) {
      RequiredPrivilege += *I;
      ++I;
      // FIXME: Simplify this logic
      if (UnMatchedPrivilege.size() > 1 && I != UnMatchedPrivilege.end()) {
        if (I == --UnMatchedPrivilege.end())
          RequiredPrivilege += " and ";
        else
          RequiredPrivilege += " , ";
      }
    }
    unsigned Warning =
        Diag->getCustomDiagID(DiagnosticsEngine::Warning,
                              "Function %0 needs additional privilege %1");
    Diag->Report(CE->getLocStart(), Warning) << FD->getNameAsString()
                                             << RequiredPrivilege;
  }
}

void CommentChecker::ReportEnumProblems(const VarDecl *VD,
                                        StringRef TypeDefName,
                                        DiagnosticsEngine *Diag) {

  // Check if API is Deprecated.
  if (IsDeprecatedFn) {
    unsigned Warning = Diag->getCustomDiagID(DiagnosticsEngine::Warning,
                                             "Enum %0 is deprecated in %1");
    Diag->Report(VD->getLocation(), Warning) << TypeDefName
                                             << ApiVersion.c_str();
    return;
  }

  // Check if API usage is valid.
  if (!IsValidAPIUsage) {
    unsigned Warning = Diag->getCustomDiagID(DiagnosticsEngine::Warning,
                                             "Enum %0 is not supported in %1");
    Diag->Report(VD->getLocation(), Warning) << TypeDefName
                                             << ApiVersion.c_str();
    return;
  }

  // Check if some privileges are missing.
  if (!UnMatchedPrivilege.empty()) {
    std::string RequiredPrivilege = "";
    for (auto I = UnMatchedPrivilege.begin(), E = UnMatchedPrivilege.end();
         I != E;) {
      RequiredPrivilege += *I;
      ++I;
      // FIXME: Simplify this logic
      if (UnMatchedPrivilege.size() > 1 && I != UnMatchedPrivilege.end()) {
        if (I == --UnMatchedPrivilege.end())
          RequiredPrivilege += " and ";
        else
          RequiredPrivilege += " , ";
      }
    }
    unsigned Warning = Diag->getCustomDiagID(
        DiagnosticsEngine::Warning, "Enum %0 needs additional privilege %1");
    Diag->Report(VD->getLocation(), Warning) << TypeDefName
                                             << RequiredPrivilege;
  }
}

// APICheckerRecursiveVisitor VisitStmt :
// .....
bool APICheckerRecursiveVisitor::VisitStmt(Stmt *S) {
  if (const CallExpr *CE = dyn_cast<CallExpr>(S)) {
    const FunctionDecl *FD = CE->getDirectCallee();
    DiagnosticsEngine *Diag = &(AstContext->getDiagnostics());
    if (FD && (FD->getBody() == NULL)) {
      FullComment *DoxyComment = AstContext->getCommentForDecl(FD, NULL);
      if (DoxyComment) {
        CommentChecker Checker(DoxyComment);
        Checker.CheckFunction();
        if (Diag)
          Checker.ReportProblems(CE, Diag);
      }
    }
  }

  return true;
}

bool APICheckerRecursiveVisitor::VisitDecl(Decl *D) {
  if (const VarDecl *VD = dyn_cast<VarDecl>(D)) {
    const Type *Ty = VD->getType().getTypePtr();
    if (!Ty)
      return true;

    if (const TypedefType *TDT = dyn_cast<TypedefType>(Ty)) {
      TypedefNameDecl *TDD = TDT->getDecl();
      StringRef TypeDefName = TDD->getName();
      CanQualType CT = AstContext->getCanonicalType(VD->getType());
      if (CT->getTypeClass() == Type::Enum) {
        const EnumType *ET = dyn_cast<EnumType>(CT->getTypePtr());
        const EnumDecl *ED = ET->getDecl();
        FullComment *DoxyComment = AstContext->getCommentForDecl(ED, NULL);
        if (DoxyComment) {
          DiagnosticsEngine *Diag = &(AstContext->getDiagnostics());
          CommentChecker Checker(DoxyComment);
          Checker.CheckFunction();
          if (Diag)
            Checker.ReportEnumProblems(VD, TypeDefName, Diag);
        }
      }
    }
  }
  return true;
}
