//===--- HTMLDiagnostics.cpp - HTML Diagnostics for Paths ----*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
//  This file defines the HTMLDiagnostics object.
//
//===----------------------------------------------------------------------===//

#include "clang/StaticAnalyzer/Core/PathDiagnosticConsumers.h"
#include "clang/AST/ASTContext.h"
#include "clang/AST/Decl.h"
#include "clang/Basic/FileManager.h"
#include "clang/Basic/SourceManager.h"
#include "clang/Lex/Lexer.h"
#include "clang/Lex/Preprocessor.h"
#include "clang/Rewrite/Core/HTMLRewrite.h"
#include "clang/Rewrite/Core/Rewriter.h"
#include "clang/StaticAnalyzer/Core/BugReporter/PathDiagnostic.h"
#include "clang/StaticAnalyzer/Core/CheckerManager.h"
#include "llvm/Support/Errc.h"
#include "llvm/Support/FileSystem.h"
#include "llvm/Support/MemoryBuffer.h"
#include "llvm/Support/Path.h"
#include "llvm/Support/raw_ostream.h"
#include <sstream>

using namespace clang;
using namespace ento;
using namespace std;
using namespace llvm;

//===----------------------------------------------------------------------===//
// Boilerplate.
//===----------------------------------------------------------------------===//

namespace {

class HTMLDiagnostics : public PathDiagnosticConsumer {
  std::string Directory;
  bool createdDir, noDir;
  const Preprocessor &PP;
  // Map of SourceManager and Html File Created
  std::map<const SourceManager *, std::string> HtmlSourceMap;
  // Map of SourceManager and Rewriter
  std::map<const SourceManager *, Rewriter *> RewriterSourceMap;

  std::string getNxtEventFile(const PathDiagnosticPiece &P);
  AnalyzerOptions &AnalyzerOpts;

public:
  HTMLDiagnostics(AnalyzerOptions &AnalyzerOpts, const std::string &prefix,
                  const Preprocessor &pp);

  ~HTMLDiagnostics() override { FlushDiagnostics(nullptr); }

  void FlushDiagnosticsImpl(std::vector<const PathDiagnostic *> &Diags,
                            FilesMade *filesMade) override;

  StringRef getName() const override { return "HTMLDiagnostics"; }

  unsigned ProcessMacroPiece(raw_ostream &os, const PathDiagnosticMacroPiece &P,
                             unsigned num);

  void HandleExternalPiece(FullSourceLoc Pos, FileID FID,
                           const PathDiagnosticPiece &P, unsigned num,
                           unsigned max, std::string prevEventFile,
                           std::string nxtEventFile);

  void HandlePiece(Rewriter &R, FileID BugFileID, const PathDiagnosticPiece &P,
                   unsigned num, unsigned max, std::string prevEventFile,
                   std::string nxtEventFile);

  void HighlightRange(Rewriter &R, FileID BugFileID, SourceRange Range,
                      const char *HighlightStart = "<span class=\"mrange\">",
                      const char *HighlightEnd = "</span>");

  void ReportDiag(const PathDiagnostic &D, FilesMade *filesMade);

  void CreateHelperHtml(Rewriter &R, std::string HtmlFileName,
                        const SourceManager *SrcMgr, FileID FID);
};

} // end anonymous namespace

HTMLDiagnostics::HTMLDiagnostics(AnalyzerOptions &AnalyzerOpts,
                                 const std::string& prefix,
                                 const Preprocessor &pp)
    : Directory(prefix), createdDir(false), noDir(false), PP(pp), AnalyzerOpts(AnalyzerOpts) {
}

void ento::createHTMLDiagnosticConsumer(AnalyzerOptions &AnalyzerOpts,
                                        PathDiagnosticConsumers &C,
                                        const std::string& prefix,
                                        const Preprocessor &PP) {
  C.push_back(new HTMLDiagnostics(AnalyzerOpts, prefix, PP));
}

//===----------------------------------------------------------------------===//
// Report processing.
//===----------------------------------------------------------------------===//

void HTMLDiagnostics::FlushDiagnosticsImpl(
  std::vector<const PathDiagnostic *> &Diags,
  FilesMade *filesMade) {
  for (std::vector<const PathDiagnostic *>::iterator it = Diags.begin(),
       et = Diags.end(); it != et; ++it) {
    ReportDiag(**it, filesMade);
  }
}

std::string HTMLDiagnostics::getNxtEventFile(const PathDiagnosticPiece &P) {

  const SourceManager *SMgr = P.getLocation().asLocation().getSourceManager();

  std::string htmlFile = HtmlSourceMap[SMgr];
  if (htmlFile.empty() == false)
    return htmlFile;

  int FD;
  SmallString<128> Model, ResultPath;
  llvm::sys::path::append(Model, Directory, "helper-%%%%%%.html");
  if (std::error_code EC =
          llvm::sys::fs::createUniqueFile(Model.str(), FD, ResultPath)) {
    llvm::errs() << "warning: could not create file in '" << Directory
                 << "': " << EC.message() << '\n';
    return "";
  }
  llvm::sys::fs::closeFileId(FD);

  string path = ResultPath.str();
  HtmlSourceMap[SMgr] = path;

  // Create a Rewriter to write to this html
  Rewriter *R =
      new Rewriter(const_cast<SourceManager &>(*SMgr), PP.getLangOpts());
  RewriterSourceMap[SMgr] = R;

  return path;
}

void HTMLDiagnostics::CreateHelperHtml(Rewriter &R, std::string HtmlFileName,
                                       const SourceManager *SrcMgr,
                                       FileID FID) {

  html::EscapeText(R, FID);
  html::AddLineNumbers(R, FID);
  // TODO: Oops using previous preprocessor.
  html::SyntaxHighlight(R, FID, PP);

  const FileEntry *Entry = SrcMgr->getFileEntryForID(FID);

  html::AddHeaderFooterInternalBuiltinCSS(R, FID, Entry->getName());

  // Get the rewrite buffer.
  const RewriteBuffer *Buf = R.getRewriteBufferFor(FID);

  if (!Buf) {
    llvm::errs() << "warning: no diagnostics generated for main file.\n";
    return;
  }

  int FD;
  SmallString<128> ResultPath;
  if (std::error_code EC =
          llvm::sys::fs::createUniqueFile(HtmlFileName, FD, ResultPath)) {
    llvm::errs() << "warning: could not create file '" << HtmlFileName
                 << "': " << EC.message() << '\n';
    return;
  }

  std::string ErrorMsg;
  llvm::raw_fd_ostream os1(FD, true);
  for (RewriteBuffer::iterator I = Buf->begin(), E = Buf->end(); I != E; ++I)
    os1 << *I;
  os1.close();
}

void HTMLDiagnostics::ReportDiag(const PathDiagnostic &D,
                                 FilesMade *filesMade) {

  // Create the HTML directory if it is missing.
  if (!createdDir) {
    createdDir = true;
    if (std::error_code ec = llvm::sys::fs::create_directories(Directory)) {
      llvm::errs() << "warning: could not create directory '"
                   << Directory << "': " << ec.message() << '\n';

      noDir = true;

      return;
    }
  }

  if (noDir)
    return;

  // Create a path for the target HTML file.
  int FD;
  SmallString<128> Model, ResultPath;

  llvm::sys::path::append(Model, Directory, "report-%%%%%%.html");

  if (std::error_code EC =
          llvm::sys::fs::createUniqueFile(Model.str(), FD, ResultPath)) {
    llvm::errs() << "warning: could not create file in '" << Directory
                 << "': " << EC.message() << '\n';
    return;
  }

  // First flatten out the entire path to make it easier to use.
  PathPieces path = D.path.flatten(/*ShouldFlattenMacros=*/false);

  // The path as already been prechecked that all parts of the path are
  // from the same file and that it is non-empty.
  const SourceManager *SMgr =
      (*(*path.begin())).getLocation().asLocation().getSourceManager();
  assert(!path.empty());
  FileID FID =
      (*path.begin())->getLocation().asLocation().getExpansionLoc().getFileID();
  assert(!FID.isInvalid());

  // Create a new rewriter to generate HTML.
  Rewriter *R =
      new Rewriter(const_cast<SourceManager &>(*SMgr), PP.getLangOpts());
  R->setFileID(FID);

  // Process the path.
  unsigned n = path.size();
  unsigned max = n;
  string filepath = ResultPath.str();
  HtmlSourceMap[SMgr] = filepath;
  RewriterSourceMap[SMgr] = R;

  std::string prevEventFile;
  std::string nxtEventFile;

  for (PathPieces::const_reverse_iterator I = path.rbegin(), E = path.rend();
       I != E; ++I, --n) {
    // We are emitting in a reverse order
    PathPieces::const_reverse_iterator End = path.rend();
    if (I != --End) {
      prevEventFile = getNxtEventFile(**(++I));
      --I;
    } else
      prevEventFile = "";
    HandlePiece(*R, FID, **I, n, max, prevEventFile, nxtEventFile);
    nxtEventFile = getNxtEventFile(**I);
  }

  // Add line numbers, header, footer, etc.

  // unsigned FID = R.getSourceMgr().getMainFileID();
  html::EscapeText(*R, FID);
  html::AddLineNumbers(*R, FID);

  // If we have a preprocessor, relex the file and syntax highlight.
  // We might not have a preprocessor if we come from a deserialized AST file,
  // for example.

  html::SyntaxHighlight(*R, FID, PP);
  html::HighlightMacros(*R, FID, PP);

  // Get the full directory name of the analyzed file.

  const FileEntry *Entry = SMgr->getFileEntryForID(FID);

  // This is a cludge; basically we want to append either the full
  // working directory if we have no directory information.  This is
  // a work in progress.

  llvm::SmallString<0> DirName;

  if (llvm::sys::path::is_relative(Entry->getName())) {
    llvm::sys::fs::current_path(DirName);
    DirName += '/';
  }

  int LineNumber =
      (*path.begin())->getLocation().asLocation().getExpansionLineNumber();
  int ColumnNumber =
      (*path.begin())->getLocation().asLocation().getExpansionColumnNumber();

  // Add the name of the file as an <h1> tag.

  {
    std::string s;
    llvm::raw_string_ostream os(s);

    os << "<!-- REPORTHEADER -->\n"
      << "<h3>Bug Summary</h3>\n<table class=\"simpletable\">\n"
          "<tr><td class=\"rowname\">File:</td><td>"
      << html::EscapeText(DirName)
      << html::EscapeText(Entry->getName())
      << "</td></tr>\n<tr><td class=\"rowname\">Location:</td><td>"
         "<a href=\"#EndPath\">line "
      << LineNumber
      << ", column "
      << ColumnNumber
      << "</a></td></tr>\n"
         "<tr><td class=\"rowname\">Description:</td><td>"
      << D.getVerboseDescription() << "</td></tr>\n";

    // Output any other meta data.

    for (PathDiagnostic::meta_iterator I=D.meta_begin(), E=D.meta_end();
         I!=E; ++I) {
      os << "<tr><td></td><td>" << html::EscapeText(*I) << "</td></tr>\n";
    }

    os << "</table>\n<!-- REPORTSUMMARYEXTRA -->\n"
          "<h3>Annotated Source Code</h3>\n";

    R->InsertTextBefore(SMgr->getLocForStartOfFile(FID), os.str());
  }

  // Embed meta-data tags.
  {
    std::string s;
    llvm::raw_string_ostream os(s);

    StringRef BugDesc = D.getVerboseDescription();
    if (!BugDesc.empty())
      os << "\n<!-- BUGDESC " << BugDesc << " -->\n";

    StringRef BugType = D.getBugType();
    if (!BugType.empty())
      os << "\n<!-- BUGTYPE " << BugType << " -->\n";

    StringRef BugCategory = D.getCategory();
    if (!BugCategory.empty())
      os << "\n<!-- BUGCATEGORY " << BugCategory << " -->\n";

    os << "\n<!-- BUGFILE " << DirName << Entry->getName() << " -->\n";

    os << "\n<!-- FILENAME " << llvm::sys::path::filename(Entry->getName())
       << " -->\n";

    // os  << "\n<!-- FUNCTIONNAME " <<  declName << " -->\n";

    os << "\n<!-- BUGLINE " << LineNumber << " -->\n";

    os << "\n<!-- BUGCOLUMN " << ColumnNumber << " -->\n";

    os << "\n<!-- BUGPATHLENGTH " << path.size() << " -->\n";

    // Mark the end of the tags.
    os << "\n<!-- BUGMETAEND -->\n";

    // Insert the text.
    R->InsertTextBefore(SMgr->getLocForStartOfFile(FID), os.str());
  }

  // Add CSS, header, and footer.

  html::AddHeaderFooterInternalBuiltinCSS(*R, FID, Entry->getName());

  // Get the rewrite buffer.
  const RewriteBuffer *Buf = R->getRewriteBufferFor(FID);

  if (!Buf) {
    llvm::errs() << "warning: no diagnostics generated for main file.\n";
    HtmlSourceMap.clear();
    RewriterSourceMap.clear();
    return;
  }

  llvm::raw_fd_ostream os(FD, true);

  if (filesMade)
    filesMade->addDiagnostic(D, getName(),
                             llvm::sys::path::filename(ResultPath));

  // Emit the HTML to disk.
  for (RewriteBuffer::iterator I = Buf->begin(), E = Buf->end(); I != E; ++I)
    os << *I;

  for (std::map<const SourceManager *, Rewriter *>::iterator it =
           RewriterSourceMap.begin();
       it != RewriterSourceMap.end(); ++it) {
    std::string HtmlFile = HtmlSourceMap[it->first];
    if (it->second->getFileID() != FID)
      CreateHelperHtml(*(it->second), HtmlFile, it->first,
                       it->second->getFileID());
  }

  HtmlSourceMap.clear();
  RewriterSourceMap.clear();
  os.close();
}

void HTMLDiagnostics::HandleExternalPiece(FullSourceLoc Pos, FileID FID,
                                          const PathDiagnosticPiece &P,
                                          unsigned num, unsigned max,
                                          std::string prevEventFile,
                                          std::string nxtEventFile) {

  // Create a new rewriter to generate HTML.
  const SourceManager *SMgr = Pos.getSourceManager();
  Rewriter *R = RewriterSourceMap[SMgr];
  if (R == NULL) {
    int FD;
    SmallString<128> Model, ResultPath;
    llvm::sys::path::append(Model, Directory, "helper-%%%%%%.html");
    if (std::error_code EC =
            llvm::sys::fs::createUniqueFile(Model.str(), FD, ResultPath)) {
      llvm::errs() << "warning: could not create file in '" << Directory
                   << "': " << EC.message() << '\n';
      return;
    }
    string path = ResultPath.str();
    HtmlSourceMap[SMgr] = path;
    // Create a Rewriter to write to this html
    R = new Rewriter(const_cast<SourceManager &>(*SMgr), PP.getLangOpts());
    RewriterSourceMap[SMgr] = R;
  }
  R->setFileID(FID);
  HandlePiece(*R, FID, P, num, max, prevEventFile, nxtEventFile);
}

void HTMLDiagnostics::HandlePiece(Rewriter &R, FileID BugFileID,
                                  const PathDiagnosticPiece &P, unsigned num,
                                  unsigned max, std::string prevEventFile,
                                  std::string nxtEventFile) {

  // For now, just draw a box above the line in question, and emit the
  // warning.
  FullSourceLoc Pos = P.getLocation().asLocation();

  if (!Pos.isValid())
    return;

  SourceManager *SM = R.getSourceManager();
  std::pair<FileID, unsigned> LPosInfo;
  if (SM == Pos.getSourceManager())
    LPosInfo = SM->getDecomposedExpansionLoc(Pos);
  else {
    LPosInfo = Pos.getSourceManager()->getDecomposedExpansionLoc(Pos);
    HandleExternalPiece(Pos, LPosInfo.first, P, num, max, prevEventFile,
                        nxtEventFile);
    return;
  }

  const llvm::MemoryBuffer *Buf = SM->getBuffer(LPosInfo.first);
  const char *FileStart = Buf->getBufferStart();

  // Compute the column number.  Rewind from the current position to the start
  // of the line.
  unsigned ColNo = SM->getColumnNumber(LPosInfo.first, LPosInfo.second);
  const char *TokInstantiationPtr = Pos.getExpansionLoc().getCharacterData();
  const char *LineStart = TokInstantiationPtr - ColNo;

  // Compute LineEnd.
  const char *LineEnd = TokInstantiationPtr;
  const char* FileEnd = Buf->getBufferEnd();
  while (*LineEnd != '\n' && LineEnd != FileEnd)
    ++LineEnd;

  // Compute the margin offset by counting tabs and non-tabs.
  unsigned PosNo = 0;
  for (const char* c = LineStart; c != TokInstantiationPtr; ++c)
    PosNo += *c == '\t' ? 8 : 1;

  // Create the html for the message.

  const char *Kind = nullptr;
  switch (P.getKind()) {
  case PathDiagnosticPiece::Call:
      llvm_unreachable("Calls should already be handled");
  case PathDiagnosticPiece::Event:  Kind = "Event"; break;
  case PathDiagnosticPiece::ControlFlow: Kind = "Control"; break;
    // Setting Kind to "Control" is intentional.
  case PathDiagnosticPiece::Macro: Kind = "Control"; break;
  }

  std::string sbuf;
  llvm::raw_string_ostream os(sbuf);

  os << "\n<tr><td class=\"num\"></td><td class=\"line\"><div id=\"";

  if (num == max)
    os << "EndPath";
  else
    os << "Path" << num;

  os << "\" class=\"msg";
  if (Kind)
    os << " msg" << Kind;
  os << "\" style=\"margin-left:" << PosNo << "ex";

  // Output a maximum size.
  if (!isa<PathDiagnosticMacroPiece>(P)) {
    // Get the string and determining its maximum substring.
    const std::string& Msg = P.getString();
    unsigned max_token = 0;
    unsigned cnt = 0;
    unsigned len = Msg.size();

    for (std::string::const_iterator I=Msg.begin(), E=Msg.end(); I!=E; ++I)
      switch (*I) {
      default:
        ++cnt;
        continue;
      case ' ':
      case '\t':
      case '\n':
        if (cnt > max_token) max_token = cnt;
        cnt = 0;
      }

    if (cnt > max_token)
      max_token = cnt;

    // Determine the approximate size of the message bubble in em.
    unsigned em;
    const unsigned max_line = 120;

    if (max_token >= max_line)
      em = max_token / 2;
    else {
      unsigned characters = max_line;
      unsigned lines = len / max_line;

      if (lines > 0) {
        for (; characters > max_token; --characters)
          if (len / characters > lines) {
            ++characters;
            break;
          }
      }

      em = characters / 2;
    }

    if (em < max_line/2)
      os << "; max-width:" << em << "em";
  }
  else
    os << "; max-width:100em";

  os << "\">";

  if (max > 1) {
    os << "<table class=\"msgT\"><tr><td valign=\"top\">";
    os << "<div class=\"PathIndex";
    if (Kind)
      os << " PathIndex" << Kind;
    os << "\">" << num << "</div>";
    size_t fileIndex = prevEventFile.find_last_of("/");
    string prevFile;
    if (fileIndex != string::npos)
      prevFile = prevEventFile.substr(fileIndex + 1);
    else
      prevFile = prevEventFile;

    if (num > 1) {
      os << "</td><td><div class=\"PathNav\"><a href=\"" << prevFile.c_str()
         << "#Path" << (num - 1) << "\" title=\"Previous event (" << (num - 1)
         << ")\">&#x2190;</a></div></td>";
    }

    os << "</td><td>";
  }

  size_t fileIndex = nxtEventFile.find_last_of("/");
  string nxtFile;
  if (fileIndex != string::npos)
    nxtFile = nxtEventFile.substr(fileIndex + 1);
  else
    nxtFile = nxtEventFile;

  if (const PathDiagnosticMacroPiece *MP =
          dyn_cast<PathDiagnosticMacroPiece>(&P)) {

    os << "Within the expansion of the macro '";

    // Get the name of the macro by relexing it.
    {
      FullSourceLoc L = MP->getLocation().asLocation().getExpansionLoc();
      assert(L.isFileID());
      StringRef BufferInfo = L.getBufferData();
      std::pair<FileID, unsigned> LocInfo = L.getDecomposedLoc();
      const char *MacroName = LocInfo.second + BufferInfo.data();
      Lexer rawLexer(SM->getLocForStartOfFile(LocInfo.first), PP.getLangOpts(),
                     BufferInfo.begin(), MacroName, BufferInfo.end());

      Token TheTok;
      rawLexer.LexFromRawLexer(TheTok);
      for (unsigned i = 0, n = TheTok.getLength(); i < n; ++i)
        os << MacroName[i];
    }

    os << "':\n";

    if (max > 1) {
      os << "</td>";
      if (num < max) {
        os << "<td><div class=\"PathNav\"><a href=\"";
        if (num == max - 1)
          os << nxtFile.c_str() << "#EndPath";
        else
          os << nxtFile.c_str() << "#Path" << (num + 1);
        os << "\" title=\"Next event (" << (num + 1)
           << ")\">&#x2192;</a></div></td>";
      }

      os << "</tr></table>";
    }

    // Within a macro piece.  Write out each event.
    ProcessMacroPiece(os, *MP, 0);
  }
  else {
    os << html::EscapeText(P.getString());

    if (max > 1) {
      os << "</td>";
      if (num < max) {
        os << "<td><div class=\"PathNav\"><a href=\"";
        if (num == max - 1)
          os << nxtFile.c_str() << "#EndPath";
        else
          os << nxtFile.c_str() << "#Path" << (num + 1);
        os << "\" title=\"Next event (" << (num + 1)
           << ")\">&#x2192;</a></div></td>";
      }

      os << "</tr></table>";
    }
  }

  os << "</div></td></tr>";

  // Insert the new html.
  unsigned DisplayPos = LineEnd - FileStart;
  SourceLocation Loc =
      SM->getLocForStartOfFile(LPosInfo.first).getLocWithOffset(DisplayPos);

  R.InsertTextBefore(Loc, os.str());

  // Now highlight the ranges.
  ArrayRef<SourceRange> Ranges = P.getRanges();
  for (ArrayRef<SourceRange>::iterator I = Ranges.begin(),
                                       E = Ranges.end(); I != E; ++I) {
    HighlightRange(R, LPosInfo.first, *I);
  }
}

static void EmitAlphaCounter(raw_ostream &os, unsigned n) {
  unsigned x = n % ('z' - 'a');
  n /= 'z' - 'a';

  if (n > 0)
    EmitAlphaCounter(os, n);

  os << char('a' + x);
}

unsigned HTMLDiagnostics::ProcessMacroPiece(raw_ostream &os,
                                            const PathDiagnosticMacroPiece& P,
                                            unsigned num) {

  for (PathPieces::const_iterator I = P.subPieces.begin(), E=P.subPieces.end();
        I!=E; ++I) {

    if (const PathDiagnosticMacroPiece *MP =
          dyn_cast<PathDiagnosticMacroPiece>(*I)) {
      num = ProcessMacroPiece(os, *MP, num);
      continue;
    }

    if (PathDiagnosticEventPiece *EP = dyn_cast<PathDiagnosticEventPiece>(*I)) {
      os << "<div class=\"msg msgEvent\" style=\"width:94%; "
            "margin-left:5px\">"
            "<table class=\"msgT\"><tr>"
            "<td valign=\"top\"><div class=\"PathIndex PathIndexEvent\">";
      EmitAlphaCounter(os, num++);
      os << "</div></td><td valign=\"top\">"
         << html::EscapeText(EP->getString())
         << "</td></tr></table></div>\n";
    }
  }

  return num;
}

void HTMLDiagnostics::HighlightRange(Rewriter& R, FileID BugFileID,
                                     SourceRange Range,
                                     const char *HighlightStart,
                                     const char *HighlightEnd) {
  SourceManager &SM = R.getSourceMgr();
  const LangOptions &LangOpts = R.getLangOpts();

  SourceLocation InstantiationStart = SM.getExpansionLoc(Range.getBegin());
  unsigned StartLineNo = SM.getExpansionLineNumber(InstantiationStart);

  SourceLocation InstantiationEnd = SM.getExpansionLoc(Range.getEnd());
  unsigned EndLineNo = SM.getExpansionLineNumber(InstantiationEnd);

  if (EndLineNo < StartLineNo)
    return;

  if (SM.getFileID(InstantiationStart) != BugFileID ||
      SM.getFileID(InstantiationEnd) != BugFileID)
    return;

  // Compute the column number of the end.
  unsigned EndColNo = SM.getExpansionColumnNumber(InstantiationEnd);
  unsigned OldEndColNo = EndColNo;

  if (EndColNo) {
    // Add in the length of the token, so that we cover multi-char tokens.
    EndColNo += Lexer::MeasureTokenLength(Range.getEnd(), SM, LangOpts)-1;
  }

  // Highlight the range.  Make the span tag the outermost tag for the
  // selected range.

  SourceLocation E =
    InstantiationEnd.getLocWithOffset(EndColNo - OldEndColNo);

  html::HighlightRange(R, InstantiationStart, E, HighlightStart, HighlightEnd);
}
