#include "llvm/Support/CommandLine.h"
#include "clang/AST/Mangle.h"
#include "clang/AST/ASTContext.h"
#include "clang/AST/ASTConsumer.h"
#include "clang/Tooling/Tooling.h"
#include "clang/Frontend/FrontendActions.h"
#include "clang/Tooling/CommonOptionsParser.h"
#include "clang/Frontend/CompilerInstance.h"
#include "clang/AST/RecursiveASTVisitor.h"
#include "llvm/Support/Debug.h"

#include <set>
#include <vector>
#include <unordered_map>

using namespace llvm;
using namespace clang;
using namespace clang::tooling;

static cl::opt<std::string> outFile("o", cl::desc("Out file name"),
                                    cl::value_desc("dest"));
static cl::opt<std::string> astFile("ast", cl::desc("AST file name"),
                                    cl::value_desc("ast"));

typedef std::set<std::string> ExtSym;
typedef std::unordered_map<std::string, std::string> FnMap;
typedef std::unordered_map<std::string, std::vector<std::string>> RevFnMap;

FnMap allData;
ExtSym extData;
bool multipleSrcFiles = false;

void serialize(raw_fd_ostream &outFs, const FnMap &allData) {
  RevFnMap mergedData;
  if (extData.size() > 0) {
    for (auto const &it : extData) {
      auto const it1 = allData.find(it);
      if (it1 == allData.end())
        continue;
      auto it2 = mergedData.find(it1->second);
      if (it2 == mergedData.end())
        it2 = mergedData.emplace(it1->second, std::vector<std::string>()).first;
      it2->second.push_back(it1->first);
    }
  } else {
    for (auto const &it1 : allData) {
      auto it2 = mergedData.find(it1.second);
      if (it2 == mergedData.end())
        it2 = mergedData.emplace(it1.second, std::vector<std::string>()).first;
      it2->second.push_back(it1.first);
    }
  }

  outFs << mergedData.size() << "\n";
  for (auto const &it : mergedData) {
    outFs << it.first << " " << it.second.size();
    for (auto const &it1 : it.second)
      outFs << " " << it1;
    outFs << "\n";
  }
}

class FNMapVisitor : public RecursiveASTVisitor<FNMapVisitor> {
public:
  FNMapVisitor(ASTContext &aCtx, MangleContext *iCntx, StringRef &infile)
      : aCtx(aCtx), ctx(iCntx), infile(infile.str()) {}

  bool shouldVisitImplicitCode() const { return true; }
  bool VisitStmt(Stmt *S) {
    if (!multipleSrcFiles)
      return false;
    if (const CallExpr *CE = dyn_cast<CallExpr>(S)) {
      if (const FunctionDecl *FD = CE->getDirectCallee()) {
        if (FD->isExternallyVisible() && FD->getBody() == NULL) {
          extData.insert(getMangledName(FD));
        }
      }
    }
    return true;
  }

  bool VisitDecl(Decl *D) {
    if (const FunctionDecl *FD = dyn_cast<FunctionDecl>(D)) {
      if (FD->isExternallyVisible() && FD->getBody() != NULL) {
        allData[getMangledName(FD)] = infile;
      }
    }
    return true;
  }

private:
  std::string getMangledName(const FunctionDecl *FD) {
    std::string mangled;
    if (!FD || !ctx)
      return mangled;

    if (const NamedDecl *D = dyn_cast<NamedDecl>(FD)) {
      raw_string_ostream stream(mangled);
      ctx->mangleNameForSA(D, stream);
      stream.flush();
    }
    return mangled;
  }

  ASTContext &aCtx;
  MangleContext *ctx;
  std::string infile;
};

class MapFunctionNamesConsumer : public ASTConsumer {
public:
  MapFunctionNamesConsumer(ASTContext &aCtx, MangleContext *iCntx,
                           StringRef &infile)
      : Visitor(FNMapVisitor(aCtx, iCntx, infile)) {}

  virtual bool HandleTopLevelDecl(DeclGroupRef DR) {
    for (auto D : DR) {
      Visitor.TraverseDecl(D);
    }
    return true;
  }

private:
  FNMapVisitor Visitor;
};

class MapFunctionNamesAction : public ASTFrontendAction {
protected:
  std::unique_ptr<ASTConsumer> CreateASTConsumer(CompilerInstance &CI,
                                                 StringRef infile) {
    MangleContext *__ItaniumCtx =
        ItaniumMangleContext::create(CI.getASTContext(), CI.getDiagnostics());
    MapFunctionNamesConsumer *PFC =
        new MapFunctionNamesConsumer(CI.getASTContext(), __ItaniumCtx, infile);
    return std::unique_ptr<ASTConsumer>(PFC);
  }
};

static cl::OptionCategory ClangFnMapGenCategory("clangFnMapGen options");
int main(int argc, const char **argv) {
  CommonOptionsParser OptionsParser(argc, argv, ClangFnMapGenCategory);

  std::vector<std::string> srcList = OptionsParser.getSourcePathList();
  ClangTool Tool(OptionsParser.getCompilations(), std::move(srcList));

  multipleSrcFiles = srcList.size() > 1;
  std::unique_ptr<FrontendActionFactory> FrontendFactory;
  FrontendFactory = newFrontendActionFactory<MapFunctionNamesAction>();
  Tool.run(FrontendFactory.get());

  std::error_code err;
  raw_fd_ostream outFs(outFile.c_str(), err, sys::fs::F_RW);
  serialize(outFs, allData);
  outFs.close();
  return 0;
}
