//===-- SocketStreamChecker.cpp ------------------------------------*- C++
//-*--//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===--------------------------------------------------------------------===//
//
// Defines a checker for proper use of socket/close APIs.
//   - If a socket has been closed with close api, it should not be accessed
//   again.
//   Accessing a closed file results in undefined behavior.
//   - If a socket was opened , it must be closed with close api before
//   the execution ends. Failing to do so results in a resource leak.
//
//===--------------------------------------------------------------------===//

#include "ClangSACheckers.h"
#include "clang/AST/Attr.h"
#include "clang/Analysis/Analyses/LiveVariables.h"
#include "clang/StaticAnalyzer/Core/BugReporter/BugType.h"
#include "clang/StaticAnalyzer/Core/Checker.h"
#include "clang/StaticAnalyzer/Core/PathSensitive/CallEvent.h"
#include "clang/StaticAnalyzer/Core/PathSensitive/CheckerContext.h"

using namespace clang;
using namespace ento;

llvm::DenseMap<const VarDecl *, bool> SVDMap;

namespace {
class FindEscaped {
public:
  llvm::SmallPtrSet<const VarDecl *, 20> Escaped;

  void operator()(const Stmt *S) {
    // Check for '&'. Any VarDecl whose address has been taken we treat as
    // escaped.
    // FIXME: What about references?
    const UnaryOperator *U = dyn_cast<UnaryOperator>(S);
    if (!U)
      return;
    if (U->getOpcode() != UO_AddrOf)
      return;

    const Expr *E = U->getSubExpr()->IgnoreParenCasts();
    if (const DeclRefExpr *DR = dyn_cast<DeclRefExpr>(E))
      if (const VarDecl *VD = dyn_cast<VarDecl>(DR->getDecl()))
        Escaped.insert(VD);
  }
};
} // end anonymous namespace

namespace {
class DeadStoreObs : public LiveVariables::Observer {
  const CFG &cfg;
  ASTContext &Ctx;
  BugReporter &BR;
  AnalysisDeclContext *AC;
  ParentMap &Parents;
  llvm::SmallPtrSet<const VarDecl *, 20> Escaped;
  const CFGBlock *currentBlock;
  enum DeadStoreKind { Standard, Enclosing, DeadIncrement, DeadInit };

public:
  DeadStoreObs(const CFG &cfg, ASTContext &ctx, BugReporter &br,
               AnalysisDeclContext *ac, ParentMap &parents,
               llvm::SmallPtrSet<const VarDecl *, 20> &escaped)
      : cfg(cfg), Ctx(ctx), BR(br), AC(ac), Parents(parents), Escaped(escaped),
        currentBlock(0) {}

  virtual ~DeadStoreObs() {}

  bool isLive(const LiveVariables::LivenessValues &Live, const VarDecl *D) {
    if (Live.isLive(D))
      return true;
    return false;
  }

  virtual void observeStmt(const Stmt *S, const CFGBlock *block,
                           const LiveVariables::LivenessValues &Live) {

    currentBlock = block;

    // Skip statements in macros.
    if (S->getLocStart().isMacroID())
      return;
    if (const DeclStmt *DS = dyn_cast<DeclStmt>(S))
      // Iterate through the decls.  Warn if any initializers are complex
      // expressions that are not live (never used).
      for (DeclStmt::const_decl_iterator DI = DS->decl_begin(),
                                         DE = DS->decl_end();
           DI != DE; ++DI) {

        VarDecl *V = dyn_cast<VarDecl>(*DI);

        if (!V)
          continue;

        if (V->hasLocalStorage()) {
          // Reference types confuse the dead stores checker.  Skip them
          // for now.
          if (V->getType()->getAs<ReferenceType>())
            return;

          if (const Expr *E = V->getInit()) {
            while (const ExprWithCleanups *exprClean =
                       dyn_cast<ExprWithCleanups>(E))
              E = exprClean->getSubExpr();

            // Don't warn on C++ objects (yet) until we can show that their
            // constructors/destructors don't have side effects.
            if (isa<CXXConstructExpr>(E))
              return;

            // A dead initialization is a variable that is dead after it
            // is initialized.  We don't flag warnings for those variables
            // marked 'unused'.
            if (!isLive(Live, V) && V->getAttr<UnusedAttr>() == 0) {
              // Special case: check for initializations with constants.
              //
              //  e.g. : int x = 0;
              //
              // If x is EVER assigned a new value later, don't issue
              // a warning.  This is because such initialization can be
              // due to defensive programming.
              if (E->isEvaluatable(Ctx))
                return;

              if (const DeclRefExpr *DRE =
                      dyn_cast<DeclRefExpr>(E->IgnoreParenCasts()))
                if (const VarDecl *VD = dyn_cast<VarDecl>(DRE->getDecl())) {
                  // Special case: check for initialization from constant
                  //  variables.
                  //
                  //  e.g. extern const int MyConstant;
                  //       int x = MyConstant;
                  //
                  if (VD->hasGlobalStorage() &&
                      VD->getType().isConstQualified())
                    return;
                  // Special case: check for initialization from scalar
                  //  parameters.  This is often a form of defensive
                  //  programming.  Non-scalars are still an error since
                  //  because it more likely represents an actual algorithmic
                  //  bug.
                  if (isa<ParmVarDecl>(VD) && VD->getType()->isScalarType())
                    return;
                }
              SVDMap[V] = true;
            }
          }
        }
      }
  }
};

} // end anonymous namespace

namespace {

// A map between socket symbol and it's state
typedef llvm::SmallVector<SymbolRef, 2> SymbolVector;

struct SocketStreamState {

private:
  enum Kind { Opened, Closed } K;
  SocketStreamState(Kind InK) : K(InK) {}

public:
  bool isOpened() const { return K == Opened; }
  bool isClosed() const { return K == Closed; }

  static SocketStreamState getOpened() { return SocketStreamState(Opened); }
  static SocketStreamState getClosed() { return SocketStreamState(Closed); }

  bool operator==(const SocketStreamState &X) const { return K == X.K; }
  void Profile(llvm::FoldingSetNodeID &ID) const { ID.AddInteger(K); }
};

class SocketStreamBugVisitor
    : public BugReporterVisitorImpl<SocketStreamBugVisitor> {

  // The allocated region tracked by the main analysis.
  SymbolRef Sym;
  bool IsLeak;

public:
  SocketStreamBugVisitor(SymbolRef S, bool isLeak = false)
      : Sym(S), IsLeak(isLeak) {}

  virtual ~SocketStreamBugVisitor() {}

  void Profile(llvm::FoldingSetNodeID &ID) const {
    static int X = 0;
    ID.AddPointer(&X);
    ID.AddPointer(Sym);
  }

  inline bool isOpened(const SocketStreamState *S,
                       const SocketStreamState *SPrev, const Stmt *Stmt) {
    return (Stmt && isa<CallExpr>(Stmt) && (S && S->isOpened()) &&
            (!SPrev || !SPrev->isOpened()));
  }

  inline bool isClosed(const SocketStreamState *S,
                       const SocketStreamState *SPrev, const Stmt *Stmt) {
    return (Stmt && isa<CallExpr>(Stmt) && (S && S->isClosed()) &&
            (!SPrev || !SPrev->isClosed()));
  }

  std::shared_ptr<clang::ento::PathDiagnosticPiece>
  VisitNode(const ExplodedNode *N, const ExplodedNode *PrevN,
            BugReporterContext &BRC, BugReport &BR);

  std::unique_ptr<PathDiagnosticPiece>
  getEndPath(BugReporterContext &BRC, const ExplodedNode *EndPathNode,
             BugReport &BR) {
    if (!IsLeak)
      return 0;

    PathDiagnosticLocation L = PathDiagnosticLocation::createEndOfPath(
        EndPathNode, *(EndPathNode->getSourceManager()));
    return llvm::make_unique<PathDiagnosticEventPiece>(
        L, BR.getDescription(), EndPathNode->getSourceManager(), false);
  }
};

class SocketStreamChecker
    : public Checker<check::PostCall, check::PreCall, check::DeadSymbols,
                     check::Bind, check::ASTCodeBody> {

  mutable std::unique_ptr<BuiltinBug> DoubleCloseBugType;
  mutable std::unique_ptr<BuiltinBug> LeakBugType;

  void reportDoubleClose(SymbolRef FileDescSym, const CallEvent &Call,
                         CheckerContext &C) const;

  void reportLeaks(SymbolVector LeakedStreams, CheckerContext &C,
                   ExplodedNode *ErrNode) const;

public:
  SocketStreamChecker();
  ~SocketStreamChecker() { SVDMap.clear(); }

  std::unique_ptr<SocketStreamBugVisitor>
  getSocketStreamBugVisitor(SymbolRef S, bool isLeak = false) const {
    return llvm::make_unique<SocketStreamBugVisitor>(S, isLeak);
  }
  /// Process socket open.
  void checkPostCall(const CallEvent &Call, CheckerContext &C) const;
  /// Process socket close.
  void checkPreCall(const CallEvent &Call, CheckerContext &C) const;

  /// If the symbol is dead check in this function if it was closed after being
  /// opened
  void checkDeadSymbols(SymbolReaper &SymReaper, CheckerContext &C) const;

  /// Deal with symbol escape as a byproduct of a bind.
  void checkBind(SVal location, SVal val, const Stmt *S,
                 CheckerContext &C) const;

  void checkASTCodeBody(const Decl *D, AnalysisManager &mgr,
                        BugReporter &BR) const;
};
}

/// The state of the checker is a map from tracked stream symbols to their
/// state. Let's store it in the ProgramState.
REGISTER_MAP_WITH_PROGRAMSTATE(StreamMap, SymbolRef, SocketStreamState)

std::shared_ptr<PathDiagnosticPiece>
SocketStreamBugVisitor::VisitNode(const ExplodedNode *N,
                                  const ExplodedNode *PrevN,
                                  BugReporterContext &BRC, BugReport &BR) {

  ProgramStateRef state = N->getState();
  ProgramStateRef statePrev = PrevN->getState();

  const SocketStreamState *RS = state->get<StreamMap>(Sym);
  const SocketStreamState *RSPrev = statePrev->get<StreamMap>(Sym);
  if (!RS)
    return 0;

  const Stmt *S = 0;
  const char *Msg = 0;
  StackHintGeneratorForSymbol *StackHint = 0;

  // Retrieve the associated statement.
  ProgramPoint ProgLoc = N->getLocation();
  if (Optional<StmtPoint> SP = ProgLoc.getAs<StmtPoint>())
    S = SP->getStmt();
  else if (Optional<CallExitEnd> Exit = ProgLoc.getAs<CallExitEnd>())
    S = Exit->getCalleeContext()->getCallSite();
  // If an assumption was made on a branch, it should be caught
  // here by looking at the state transition.
  else if (Optional<BlockEdge> Edge = ProgLoc.getAs<BlockEdge>()) {
    const CFGBlock *srcBlk = Edge->getSrc();
    S = srcBlk->getTerminator();
  }
  if (!S)
    return 0;

  if (isOpened(RS, RSPrev, S)) {
    Msg = "Socket is opened";
    StackHint = new StackHintGeneratorForSymbol(Sym, "Opened Socket");
  } else if (isClosed(RS, RSPrev, S)) {
    Msg = "Socket is closed";
    StackHint = new StackHintGeneratorForSymbol(Sym, "Closed Socket");
  }

  if (!Msg)
    return 0;
  assert(StackHint);

  // Generate the extra diagnostic.
  PathDiagnosticLocation Pos(S, *(N->getSourceManager()),
                             N->getLocationContext());
  return std::make_shared<PathDiagnosticEventPiece>(
      Pos, Msg, N->getSourceManager(), true, StackHint);
}

SocketStreamChecker::SocketStreamChecker() {
  // Initialize the bug types.
  DoubleCloseBugType.reset(new BuiltinBug(this, "Double close"));

  LeakBugType.reset(new BuiltinBug(this, "Resource Leak"));
  // Sinks are higher importance bugs as well as calls to assert() or exit(0).
  LeakBugType->setSuppressOnSink(false);
}

void SocketStreamChecker::checkASTCodeBody(const Decl *D, AnalysisManager &mgr,
                                           BugReporter &BR) const {
  if (const FunctionDecl *FD = dyn_cast<FunctionDecl>(D))
    if (FD->isTemplateInstantiation())
      return;

  if (LiveVariables *L = mgr.getAnalysis<LiveVariables>(D)) {
    CFG &cfg = *mgr.getCFG(D);
    AnalysisDeclContext *AC = mgr.getAnalysisDeclContext(D);
    ParentMap &pmap = mgr.getParentMap(D);
    FindEscaped FS;
    cfg.VisitBlockStmts(FS);
    DeadStoreObs A(cfg, BR.getContext(), BR, AC, pmap, FS.Escaped);
    L->runOnAllBlocks(A);
  }
}

void SocketStreamChecker::checkBind(SVal location, SVal val, const Stmt *S,
                                    CheckerContext &C) const {
  SymbolRef FileID = val.getAsSymbol();
  ProgramStateRef State = C.getState();
  const SocketStreamState *SS = State->get<StreamMap>(FileID);
  if (!SS)
    return;
  if (!SS->isOpened())
    return;
  if (const VarRegion *LVR =
          dyn_cast_or_null<VarRegion>(location.getAsRegion())) {
    const VarDecl *VD = LVR->getDecl();
    llvm::DenseMap<const VarDecl *, bool>::iterator DeadVar = SVDMap.find(VD);
    if (DeadVar != SVDMap.end()) {
      if (DeadVar->second) {
        ExplodedNode *ErrNode = C.addTransition(State);
        SmallString<200> buf;
        llvm::raw_svector_ostream os(buf);
        os << " Opened socket is never closed; potential resource leak";
        if (VD) {
          os << " of resource pointed to by '";
          os << VD->getName();
          os << '\'';
        }
        auto R = llvm::make_unique<BugReport>(*LeakBugType, os.str(), ErrNode);
        R->markInteresting(FileID);
        C.emitReport(std::move(R));
        SVDMap.erase(DeadVar);
        return;
      }
    }
  }
}

void SocketStreamChecker::checkPostCall(const CallEvent &Call,
                                        CheckerContext &C) const {

  const FunctionDecl *FD = dyn_cast_or_null<FunctionDecl>(Call.getDecl());
  if (!FD)
    return;

  if (!Call.isGlobalCFunction("socket") || FD->getNameAsString() != "socket")
    return;

  // Get the symbolic value corresponding to the socket handle.
  SymbolRef SocketDesc = Call.getReturnValue().getAsSymbol();
  if (!SocketDesc)
    return;

  // Generate the next transition (an edge in the exploded graph).
  ProgramStateRef State = C.getState();
  State = State->set<StreamMap>(SocketDesc, SocketStreamState::getOpened());
  C.addTransition(State);
}

void SocketStreamChecker::checkPreCall(const CallEvent &Call,
                                       CheckerContext &C) const {

  const FunctionDecl *FD = dyn_cast_or_null<FunctionDecl>(Call.getDecl());
  if (!FD)
    return;

  if (!Call.isGlobalCFunction("close") || FD->getNameAsString() != "close")
    return;

  if (Call.getNumArgs() != 1)
    return;

  // Get the symbolic value corresponding to the file handle.
  SymbolRef SocketDesc = Call.getArgSVal(0).getAsSymbol();
  if (!SocketDesc)
    return;

  // Check if the stream has already been closed.
  ProgramStateRef State = C.getState();
  const SocketStreamState *SS = State->get<StreamMap>(SocketDesc);
  if (SS && SS->isClosed()) {
    reportDoubleClose(SocketDesc, Call, C);
    return;
  }

  // Generate the next transition, in which the stream is closed.
  State = State->set<StreamMap>(SocketDesc, SocketStreamState::getClosed());
  C.addTransition(State);
}

static bool isLeaked(SymbolRef Sym, const SocketStreamState &SS, bool IsSymDead,
                     ProgramStateRef State) {
  if (IsSymDead && SS.isOpened()) {

    ConstraintManager &CMgr = State->getConstraintManager();
    const llvm::APSInt *isOpenSuccess = CMgr.getSymVal(State, Sym);
    if (isOpenSuccess) {
      // [TODO]
      // -1 rotates to UINT_MAX. How to reteive int value from APSInt?
      // This method of using toString may not be efficient
      std::string str = isOpenSuccess->toString(16);
      return !strcmp(str.c_str(), "0xffffffff");
    }
    // We have a symbol not translated to APSInt.Socket open passed.
    return true;
  }
  return false;
}

void SocketStreamChecker::checkDeadSymbols(SymbolReaper &SymReaper,
                                           CheckerContext &C) const {
  ProgramStateRef State = C.getState();
  SymbolVector LeakedStreams;
  StreamMapTy TrackedStreams = State->get<StreamMap>();
  for (StreamMapTy::iterator I = TrackedStreams.begin(),
                             E = TrackedStreams.end();
       I != E; ++I) {
    SymbolRef Sym = I->first;
    bool IsSymDead = SymReaper.isDead(Sym);

    // Collect leaked symbols.
    if (isLeaked(Sym, I->second, IsSymDead, State))
      LeakedStreams.push_back(Sym);

    // Remove the dead symbol from the streams map.
    if (IsSymDead)
      State = State->remove<StreamMap>(Sym);
  }

  ExplodedNode *N = C.addTransition(State);
  reportLeaks(LeakedStreams, C, N);
}

// Report Erros

// Handling double close of sockets

void SocketStreamChecker::reportDoubleClose(SymbolRef FileDescSym,
                                            const CallEvent &Call,
                                            CheckerContext &C) const {
  // We reached a bug, stop exploring the path here by generating a sink.
  ExplodedNode *ErrNode = C.addTransition(C.getState());
  // If we've already reached this node on another path, return.
  if (!ErrNode)
    return;

  // Generate the report.
  auto R = llvm::make_unique<BugReport>(
      *DoubleCloseBugType, "Closing a previously closed socket stream",
      ErrNode);
  R->addRange(Call.getSourceRange());
  R->markInteresting(FileDescSym);
  R->addVisitor(getSocketStreamBugVisitor(FileDescSym));
  C.emitReport(std::move(R));
}

void SocketStreamChecker::reportLeaks(SymbolVector LeakedStreams,
                                      CheckerContext &C,
                                      ExplodedNode *ErrNode) const {
  // Attach bug reports to the leak node.
  // TODO: Identify the leaked file descriptor.
  for (llvm::SmallVector<SymbolRef, 2>::iterator I = LeakedStreams.begin(),
                                                 E = LeakedStreams.end();
       I != E; ++I) {
    auto R = llvm::make_unique<BugReport>(
        *LeakBugType, "Opened socket is never closed; potential resource leak",
        ErrNode);
    R->markInteresting(*I);
    R->addVisitor(getSocketStreamBugVisitor(*I, true));
    C.emitReport(std::move(R));
  }
}

void ento::registerSocketStreamChecker(CheckerManager &mgr) {
  mgr.registerChecker<SocketStreamChecker>();
}
