//===--- XMLDiagnostics.cpp - Helper classes for Cross TU analysis ---* C++
//*===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
//  This file defines Helper classes for Cross TU analysis.
//
//===----------------------------------------------------------------------===//

#include "XTUAnalysisHelper.h"
#include "clang/AST/ASTContext.h"
#include "clang/AST/ASTImporter.h"
#include "clang/AST/GlobalDecl.h"
#include "clang/AST/Mangle.h"
#include "clang/Basic/Diagnostic.h"
#include "clang/Frontend/ASTUnit.h"
#include "clang/Frontend/CompilerInstance.h"
#include "clang/Frontend/FrontendAction.h"
#include "llvm/ADT/StringRef.h"
#include "llvm/Support/Debug.h"
#include "llvm/Support/Path.h"
#include "llvm/Support/raw_ostream.h"

#include <fstream>
#include <set>
#include <unordered_map>

using namespace llvm;
using namespace clang;

class FdData {
public:
  FdData(const std::string *astName = nullptr)
      : isFnDecl(false), data(astName) {}

  bool isResolved() { return isFnDecl; }

  void setAstName(const std::string *astName) {
    isFnDecl = false;
    data = astName;
  }
  void setFnDecl(const FunctionDecl *fd) {
    isFnDecl = true;
    data = fd;
  }

  const std::string getAstName() {
    assert(!isFnDecl);
    return *(const std::string *)data;
  }
  const FunctionDecl *getFnDecl() {
    assert(isFnDecl);
    return (const FunctionDecl *)data;
  }

  bool isFnDecl;
  const void *data;
};

std::unique_ptr<CompilerInstance> Clang(new CompilerInstance());
CompilerInstance &CI = *Clang.get();
typedef std::set<std::string> XtuAstNames;
typedef std::unordered_map<std::string, FdData> XtuFdMap;
static XtuFdMap deserialize(std::ifstream &inFs) {
  XtuFdMap data;
  size_t astCnt = 0;
  static XtuAstNames astNames;
  while (inFs >> astCnt) {
    for (size_t i = 0; i < astCnt; ++i) {
      std::string astName, sym;
      inFs >> astName;
      auto it = astNames.insert(astName);
      const std::string &val = *it.first;
      size_t symCnt;
      inFs >> symCnt;
      for (size_t j = 0; j < symCnt; ++j) {
        inFs >> sym;
        data[sym] = FdData(&val);
      }
    }
  }
  return data;
}

static XtuFdMap getExternMapData() {
  SmallString<128> currDir;
  sys::fs::current_path(currDir);
  while (!currDir.empty()) {
    SmallString<128> mapFile(currDir);
    sys::path::append(mapFile, "externalFnMap.txt");
    std::ifstream inFs(mapFile.str(), std::ios_base::in);
    if (inFs.good()) {
      return deserialize(inFs);
    }

    currDir = sys::path::parent_path(currDir);
  }
  assert(false && "Unable to find fn map file");
  return XtuFdMap();
}

static bool xtuAnalysis = false;
static SmallString<128> mainfile;
static std::string mainAstfile;
bool XtuHelper::isXtuAnalysis() { return xtuAnalysis; }

static StringRef getMainFile() { return mainfile; }
static void setMainFile(const NamedDecl *ND) {
  if (!mainfile.empty() || !ND)
    return;

  const SourceManager &SM = ND->getASTContext().getSourceManager();
  const FileEntry *Entry = SM.getFileEntryForID(SM.getMainFileID());
  mainfile = Entry->getName();
  sys::fs::make_absolute(mainfile);
}

typedef std::unordered_map<std::string, const ASTUnit *> XtuAstMap;

const ASTUnit *getAstUnit(std::string astFileName,
                          IntrusiveRefCntPtr<DiagnosticsEngine> &Diags,
                          PCHContainerReader *PCHContainerRdr) {
  static XtuAstMap ASTMap;
  auto au = ASTMap.find(astFileName);
  if (au != ASTMap.end())
    return au->second;

  FileSystemOptions FileSystemOpts;
  std::unique_ptr<ASTUnit> unit = ASTUnit::LoadFromASTFile(
      astFileName, *PCHContainerRdr, Diags, FileSystemOpts);
  if (unit) {
    if (AnalyzerOptions::EmitAstDeps) {
      const SourceManager &SM = unit->getASTContext().getSourceManager();
      const FileEntry *Entry = SM.getFileEntryForID(SM.getMainFileID());
      if (!getMainFile().equals(Entry->getName())) {
        dbgs() << "Importing AST for File: " << Entry->getName() << "\n";
      }
    }
    ASTUnit *astUnit = unit.release();
    ASTMap[astFileName] = astUnit;
    xtuAnalysis = true;
    return astUnit;
  }
  return nullptr;
}

std::string getMangledName(const FunctionDecl *FD, DiagnosticsEngine &Diags) {
  std::string mangled;
  if (!FD)
    return mangled;
  setMainFile(FD);

  std::unique_ptr<MangleContext> ctx(
      ItaniumMangleContext::create(FD->getASTContext(), Diags));
  if (ctx) {
    if (const NamedDecl *D = dyn_cast<NamedDecl>(FD)) {
      raw_string_ostream stream(mangled);
      ctx->mangleNameForSA(D, stream);
      stream.flush();
    }
  }
  return mangled;
}

const FunctionDecl *
XtuHelper::findFunctionDeclInExternalAST(const FunctionDecl *FD,
                                         PCHContainerReader *PCHContainerRdr) {
  static XtuFdMap FdMap = getExternMapData();
  if (!FD->getIdentifier())
    return FD;

  IntrusiveRefCntPtr<DiagnosticsEngine> Diags =
      CompilerInstance::createDiagnostics(new DiagnosticOptions);
  std::string mangledName(getMangledName(FD, *Diags));
  const auto &fdItr = FdMap.find(mangledName);
  if (fdItr == FdMap.end())
    return FD;
  if (!fdItr->second.isResolved()) {
    const ASTUnit *unit =
        getAstUnit(fdItr->second.getAstName(), Diags, PCHContainerRdr);
    if (!unit)
      return FD;

    TranslationUnitDecl *TU = unit->getASTContext().getTranslationUnitDecl();
    for (auto const D : TU->decls()) {
      const FunctionDecl *ND = dyn_cast<FunctionDecl>(D);
      if (ND && ND->hasBody()) {
        std::string ndMangledName(getMangledName(ND, *Diags));
        if (ndMangledName.empty())
          continue;
        const auto &ndItr = FdMap.find(ndMangledName);
        if (ndItr == FdMap.end())
          continue;
        ndItr->second.setFnDecl(ND);
        assert(ndItr->second.isResolved());
      }
    }
  }
  assert(fdItr->second.isResolved());
  return fdItr->second.getFnDecl();
}
