"""
Snapdiff is used to generate image and repo diffs. You can use it as a module:

   >>> import snapdiff
   >>> snapdiff.diff_to_json('old url', 'new url', style='repo')
   >>> snapdiff.diff_to_html('old url', 'new url', style='image')

or use it directly:
    snap-diff [-h] [--json] [-t] old new

    Diff two repos with different urls

    positional arguments:
      old         old repo
      new         new repo

    optional arguments:
      -h, --help  show this help message and exit
      --json      output json diffs
      -t          which diff you want(repo | image, default is repo)
      ...

"""

__title__ = 'python-snapdiff'
__version__ = '0.1'

from itertools import izip_longest
import json
import os
import shutil
import re
import subprocess
import io

import yaml

from .image import ks_diff, packages
from .render import output_html
from .repo import Repo

def escape_html(s):
    s = s.replace("&", "&amp;");
    s = s.replace("<", "&lt;");
    s = s.replace(">", "&gt;");
    s = s.replace("'", "&#39;");
    s = s.replace('"', "&quot;");
    return s

def escape_html_newline(s):
    s = s.replace('\n', "<br/>");
    return s

def diff_to_json(old_url, new_url, **kwargs):
    """Output diffs' json format.

    :param old_url: old repo or image url
    :param new_url: new repo or image url
    :param style: (optional) repo or image type, default is repo
    """

    if not old_url or not new_url:
        return

    style = kwargs.get('style') or 'repo'

    if style == 'repo':
        old, new = Repo(old_url).packages, Repo(new_url).packages
    elif style == 'image':
        old, new = packages(old_url), packages(new_url)
    else:
        return

    added, removed, modified, rebuilded = [], [], [], []
    package_names = set(old.__dict__.keys() + new.__dict__.keys())

    def _get_commit_log_delta(old_proj, new_proj, old_commit_id, new_commit_id):
        cache_dir = os.getenv('GIT_CACHE_DIR')
        git_dir = os.path.join(cache_dir, new_proj)
        log_filename = 'git_log'

        if not os.path.exists(git_dir):
            return []

        revision_range = ''
        if old_commit_id == None or len(old_commit_id) == 0:
            revision_range = new_commit_id
        else:
            revision_range = old_commit_id + ".." + new_commit_id

        if old_proj != new_proj:
            revision_range = new_commit_id

        f = io.open(log_filename, 'w', encoding='utf-8', errors='ignore')
        subprocess.call(['git', '--git-dir='+git_dir+'.git', 'log',
                        revision_range,
                        '--pretty="format:||||%H|||%cN<%cE>|||%cD|||%s%n%b||||"'], stdout=f)
                        #'--pretty="format:||||%H|||%aN<%aE>%n%s%n%b||||"'], stdout=f)
        f.close()
        f = io.open(log_filename, 'r', encoding='utf-8', errors='ignore')
        contents = escape_html(f.read())
        sp = contents.split("||||")
        data = []
        for idx in range(1, len(sp), 2):
            l = sp[idx].split("|||")
            data.append({"commit_id": l[0],
                         "committer": l[1],
                         "commit_date": l[2],
                         "commit_log": {'html':escape_html_newline(l[3]), 'raw':l[3]}})
        f.close()
        return data

    def _get_git_web(url):
        matchdomain = re.match(r"http[s]?://[a-z0-9\.]*", url)
        if not matchdomain:
            return "#"
        srcdomain = matchdomain.group(0)
        profile = '/etc/%s/profile.yaml' % __title__
        gitweb_dict = {}
        if not os.path.exists(profile):
            return "#"
        with open(profile, 'r') as pfile:
            gitweb_dict = yaml.load(pfile)
        for domain in gitweb_dict['profile']:
            if domain['name'] == srcdomain:
                return domain['gitweb']
        return "#"

    def _pair_old_new():
        """Generate old and new pkg pair"""
        for name in package_names:
            if old[name] is None:
                for pkg in new[name]:
                    yield (None, pkg)
            elif new[name] is None:
                for pkg in old[name]:
                    yield (pkg, None)
            else:
                for old_pkg in old[name]:
                    for new_pkg in new[name]:
                        if old_pkg.version.ver == new_pkg.version.ver:
                            yield (old_pkg, new_pkg)
                            old[name].remove(old_pkg)
                            new[name].remove(new_pkg)
                            break
                for pair in izip_longest(old[name], new[name]):
                    yield pair
    for old_pkg, new_pkg in _pair_old_new():
        if old_pkg is None:
            added.append(new_pkg)
        elif new_pkg is None:
            removed.append(old_pkg)
        elif old_pkg.version.vcs != new_pkg.version.vcs or \
                old_pkg.version.ver != new_pkg.version.ver:
            modified.append((old_pkg, new_pkg))
        elif old_pkg.version.rel != new_pkg.version.rel:
            rebuilded.append((old_pkg, new_pkg))

    obj = {style: {'old': old_url, 'new': new_url},
            'diff': {
              'added': [],
              'removed': [],
              'modified': [],
              'rebuilded': [],
            }}
    #-----------------------------------------------------------------------------
    added_p = {}
    for _new in added:
        _git_path = _new.version.vcs.split('#', 1)[0]
        _version = '-'.join([_new.version.ver, _new.version.rel])

        if _git_path not in added_p:
            added_p[_git_path] = {}

        if _version not in added_p[_git_path]:
            added_p[_git_path][_version] = []

        added_p[_git_path][_version].append(_new)

    for _git_path in added_p:
        for _version in added_p[_git_path]:
            _p = added_p[_git_path][_version][0]
            _commit_id = _p.version.vcs.split('#',1)[1]
            _codediff = None
            _commit_log = _get_commit_log_delta('', _git_path, '', _commit_id)

            ps = []
            for _pp in added_p[_git_path][_version]:
                ps.append('.'.join([_pp.name,_pp.arch]))

            p = {'oldpkg': None,
                 'newpkg': {
                     'name': ','.join(ps),
                     'version': {
                         'epoch': _p.version.epoch,
                         'rel': _p.version.rel,
                         'ver': _p.version.ver,
                     },
                 'git_path': _git_path,
                 'commit_id': _commit_id
                 },
                 'codediff': _codediff,
                 'commit_log': _commit_log
                }
            obj['diff']['added'].append(p)

    #-----------------------------------------------------------------------------
    removed_p = {}
    for _old in removed:
        _git_path = _old.version.vcs.split('#', 1)[0]
        _version = '-'.join([_old.version.ver, _old.version.rel])

        if _git_path not in removed_p:
            removed_p[_git_path] = {}

        if _version not in removed_p[_git_path]:
            removed_p[_git_path][_version] = []

        removed_p[_git_path][_version].append(_old)

    for _git_path in removed_p:
        for _version in removed_p[_git_path]:
            _p = removed_p[_git_path][_version][0]
            _commit_id = _p.version.vcs.split('#',1)[1]
            _codediff = None

            ps = []
            for _pp in removed_p[_git_path][_version]:
                ps.append('.'.join([_pp.name,_pp.arch]))

            obj['diff']['removed'].append(
                    {'oldpkg': {
                        'name': ','.join(ps),
                        'version': {
                            'epoch': _p.version.epoch,
                            'rel': _p.version.rel,
                            'ver': _p.version.ver,
                            },
                        'git_path': _git_path,
                        'commit_id': _commit_id
                        },
                        'newpkg': None,
                        'codediff': _codediff,
                        })

    #-----------------------------------------------------------------------------
    modified_p = {}
    for _old, _new in modified:
        _git_path = _old.version.vcs.split('#',1)[0]
        _version = '-'.join([_old.version.ver, _old.version.rel, _new.version.ver, _new.version.rel])

        if _git_path not in modified_p:
            modified_p[_git_path] = {}

        if _version not in modified_p:
            modified_p[_git_path][_version] = []

        modified_p[_git_path][_version].append((_old,_new))

    for _git_path in modified_p:
        for _version in modified_p[_git_path]:
            (_old,_new) = modified_p[_git_path][_version][0]
            _old_git_path  = _old.version.vcs.split('#', 1)[0]
            _old_commit_id = _old.version.vcs.split('#', 1)[1]
            _new_git_path  = _new.version.vcs.split('#', 1)[0]
            _new_commit_id = _new.version.vcs.split('#', 1)[1]

            _codediff = None
            _commit_log = _get_commit_log_delta(_old_git_path, _new_git_path,
                                                _old_commit_id, _new_commit_id)

            ps = []
            for _old, _new in modified_p[_git_path][_version]:
                ps.append('.'.join([_old.name,_old.arch]))

            names = ','.join(ps)
            obj['diff']['modified'].append(
                    {'oldpkg': {
                        'name': names,
                        'version': {
                            'epoch': _old.version.epoch,
                            'rel': _old.version.rel,
                            'ver': _old.version.ver,
                            },
                        'git_path': _old_git_path,
                        'commit_id': _old_commit_id
                        },
                        'newpkg':{
                            'name': names,
                            'version': {
                                'epoch': _new.version.epoch,
                                'rel': _new.version.rel,
                                'ver': _new.version.ver,
                                },
                            'git_path': _new_git_path,
                            'commit_id': _new_commit_id
                            },
                        'codediff': _codediff,
                        'commit_log': _commit_log
                        })

    #-----------------------------------------------------------------------------
    rebuilded_p = {}
    for _old, _new in rebuilded:
        _git_path = _old.version.vcs.split('#',1)[0]
        _version = '-'.join([_old.version.ver, _old.version.rel, _new.version.ver, _new.version.rel])

        if _git_path not in rebuilded_p:
            rebuilded_p[_git_path] = {}

        if _version not in rebuilded_p:
            rebuilded_p[_git_path][_version] = []

        rebuilded_p[_git_path][_version].append((_old,_new))

    for _git_path in rebuilded_p:
        for _version in rebuilded_p[_git_path]:
            (_old,_new) = rebuilded_p[_git_path][_version][0]
            _old_git_path  = _old.version.vcs.split('#', 1)[0]
            _old_commit_id = _old.version.vcs.split('#', 1)[1]
            _new_git_path  = _new.version.vcs.split('#', 1)[0]
            _new_commit_id = _new.version.vcs.split('#', 1)[1]

            _codediff = None

            ps = []
            for _old, _new in rebuilded_p[_git_path][_version]:
                ps.append('.'.join([_old.name,_old.arch]))

            names = ','.join(ps)
            obj['diff']['rebuilded'].append(
                    {'oldpkg': {
                        'name': names,
                        'version': {
                            'epoch': _old.version.epoch,
                            'rel': _old.version.rel,
                            'ver': _old.version.ver,
                            },
                        'git_path': _old_git_path,
                        'commit_id': _old_commit_id
                        },
                        'newpkg':{
                            'name': names,
                            'version': {
                                'epoch': _new.version.epoch,
                                'rel': _new.version.rel,
                                'ver': _new.version.ver,
                                },
                            'git_path': _new_git_path,
                            'commit_id': _new_commit_id
                            },
                        'codediff': _codediff,
                        })

    #-----------------------------------------------------------------------------
    #obj = {style: {'old': old_url, 'new': new_url},
            #'diff': {
                #'added': [
                    #{'oldpkg': None,
                    #'newpkg': {
                        #'name': _new.name,
                        #'version': {
                            #'epoch': _new.version.epoch,
                            #'rel': _new.version.rel,
                            #'ver': _new.version.ver,
                            #},
                        ## we use '#' to split vcs into git_path and commit_id,
                        ## but commit_id maybe contains '#', so split the 1st.
                        #'git_path': _new.version.vcs.split('#', 1)[0],
                        #'commit_id': _new.version.vcs.split('#', 1)[1],
                        #},
                    #'codediff': None,
                    #'commit_log': _get_commit_log_delta('', _new.version.vcs.split('#',1)[0],
                                                        #'', _new.version.vcs.split('#',1)[1])
                    #} for _new in added],
                #'removed': [
                    #{'oldpkg': {
                        #'name': _old.name,
                        #'version': {
                            #'epoch': _old.version.epoch,
                            #'rel': _old.version.rel,
                            #'ver': _old.version.ver,
                            #},
                        #'git_path': _old.version.vcs.split('#', 1)[0],
                        #'commit_id': _old.version.vcs.split('#', 1)[1],
                        #},
                    #'newpkg': None,
                    #'codediff': None,
                    #} for _old in removed],
                #'modified': [
                    #{'oldpkg': {
                        #'name': _old.name,
                        #'version': {
                            #'epoch': _old.version.epoch,
                            #'rel': _old.version.rel,
                            #'ver': _old.version.ver,
                            #},
                        #'git_path': _old.version.vcs.split('#', 1)[0],
                        #'commit_id': _old.version.vcs.split('#', 1)[1],
                        #},
                    #'newpkg': {
                        #'name': _new.name,
                        #'version': {
                            #'epoch': _new.version.epoch,
                            #'rel': _new.version.rel,
                            #'ver': _new.version.ver,
                            #},
                        #'git_path': _new.version.vcs.split('#', 1)[0],
                        #'commit_id': _new.version.vcs.split('#', 1)[1],
                        #},
                    #'codediff': None,
                    #'commit_log': _get_commit_log_delta(_old.version.vcs.split('#',1)[0],
                                                        #_new.version.vcs.split('#',1)[0],
                                                        #_old.version.vcs.split('#',1)[1],
                                                        #_new.version.vcs.split('#',1)[1]),
                    #} for _old, _new in modified],
                 #'rebuilded': [{
                     #'oldpkg': {
                         #'name': _old.name,
                         #'version': {
                             #'epoch': _old.version.epoch,
                             #'rel': _old.version.rel,
                             #'ver': _old.version.ver,
                             #},
                        #'git_path': _old.version.vcs.split('#', 1)[0],
                        #'commit_id': _old.version.vcs.split('#', 1)[1],
                        #},
                     #'newpkg': {
                         #'name': _new.name,
                         #'version': {
                             #'epoch': _new.version.epoch,
                             #'rel': _new.version.rel,
                             #'ver': _new.version.ver,
                             #},
                         #'git_path': _new.version.vcs.split('#', 1)[0],
                         #'commit_id': _new.version.vcs.split('#', 1)[1],
                         #},
                     #'codediff': None,
                     #} for _old, _new in rebuilded],
                #}
            #}

    if style == 'image':
        obj['diff']['ks'] = ks_diff(old_url, new_url)

    obj['gitweb'] = {'old': _get_git_web(old_url),
                    'new': _get_git_web(new_url)}

    return json.dumps(obj, indent=4)

def diff_to_csv(old_url, new_url, **kwargs):
    """Output diffs' html format.

    :param old_url: old repo or image url
    :param new_url: new repo or image url
    :param style: repo or image type, default is repo
    """

    style = kwargs.get('style') or 'repo'
    diff_name = kwargs.get('diff_name') or 'diff'

    json_obj = diff_to_json(old_url, new_url, style=style)

    if json_obj is None:
        return

    data = json.loads(json_obj)

    context = {
        'old': {'url': old_url,
            'name': re.search(r'\w*_\d{8}\.\d*', old_url).group(0)},
        'new': {'url': new_url,
            'name': re.search(r'\w*_\d{8}\.\d*', new_url).group(0)},
        'diff': data['diff'],
        'filename': diff_name,
        'gitweb': data['gitweb']
        }
    if style == 'repo':
        return output_html('diff.csv', **context) # pylint: disable=W0142
    elif style == 'image':
        return output_html('image_diff.csv', **context) # pylint: disable=W0142

def diff_to_html(old_url, new_url, **kwargs):
    """Output diffs' html format.

    :param old_url: old repo or image url
    :param new_url: new repo or image url
    :param style: repo or image type, default is repo
    """

    style = kwargs.get('style') or 'repo'
    diff_name = kwargs.get('diff_name') or 'diff'

    json_obj = diff_to_json(old_url, new_url, style=style)

    if json_obj is None:
        return

    data = json.loads(json_obj)

    context = {
        'old': {'url': old_url,
            'name': re.search(r'\w*_\d{8}\.\d*', old_url).group(0)},
        'new': {'url': new_url,
            'name': re.search(r'\w*_\d{8}\.\d*', new_url).group(0)},
        'diff': data['diff'],
        'filename': diff_name,
        'gitweb': data['gitweb']
        }
    if style == 'repo':
        return output_html('diff.html', **context) # pylint: disable=W0142
    elif style == 'image':
        return output_html('image_diff.html', **context) # pylint: disable=W0142

def diff_to_dist(old_url, new_url, dist_path, **kwargs):
    """Create a dist-dir of diffs, contains html and css.

    :param old_url: old repo or image url
    :param new_url: new repo or image url
    :param dist_path: where you put diff
    :param diff_name: (optional) the diff file name
    :param style: (optional) repo or image type, default is repo
    """
    static_dir = os.path.join(
        os.path.dirname(os.path.abspath(__file__)), 'static')

    if not os.path.exists(dist_path):
        os.makedirs(dist_path)

    for root, _dirs, files in os.walk(static_dir):
        for filename in files:
            if filename.endswith('.css'):
                if not os.path.exists(os.path.join(dist_path, 'css')):
                    os.makedirs(os.path.join(dist_path, 'css'))
                shutil.copy(os.path.join(root, filename),
                    os.path.join(dist_path, 'css', filename))
            elif filename.endswith('.png'):
                if not os.path.exists(os.path.join(dist_path, 'img')):
                    os.makedirs(os.path.join(dist_path, 'img'))
                shutil.copy(os.path.join(root, filename),
                    os.path.join(dist_path, 'img', filename))

    diff_name = kwargs.get('diff_name') or 'diff'

    output = diff_to_html(old_url, new_url, style=kwargs.get('style'), diff_name=diff_name)
    with io.open(os.path.join(dist_path, '%s.html' % diff_name), 'w', encoding='utf-8', errors='ignore') as fname:
        fname.write(output)

    output = diff_to_csv(old_url, new_url, style=kwargs.get('style'), diff_name=diff_name)
    with io.open(os.path.join(dist_path, '%s.csv' % diff_name), 'w', encoding='utf-8', errors='ignore') as fname:
        fname.write(output)
    return dist_path
