# vim: set fileencoding=utf-8 :
#
# (C) 2006-2011 Guido Guenther <agx@sigxcpu.org>
# (C) 2012 Intel Corporation <markus.lehtonen@linux.intel.com>
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
"""run commands to build an RPM package out of a git repository"""

import ConfigParser
import errno
import os, os.path
import sys
import shutil
import re
import gbp.rpm as rpm
from gbp.command_wrappers import (Command,
                                  RunAtCommand, CommandExecFailed,
                                  RemoveTree)
from gbp.config import (GbpOptionParserRpm, GbpOptionGroup)
from gbp.rpm.git import (GitRepositoryError, RpmGitRepository)
from gbp.errors import GbpError
import gbp.log
import gbp.notifications
from gbp.utils import TempDir
from gbp.scripts.common.buildpackage import (index_name, wc_names,
                                             git_archive_submodules,
                                             git_archive_single, dump_tree,
                                             write_wc, drop_index)
from gbp.pkg import (compressor_opts, compressor_aliases)

def git_archive(repo, spec, output_dir, treeish, comp_type, comp_level, with_submodules):
    "create a compressed orig tarball in output_dir using git_archive"
    comp_opts = ''
    if spec.orig_format == 'tar':
        try:
            comp_opts = compressor_opts[comp_type][0]
        except KeyError:
            raise GbpError, "Unsupported compression type '%s'" % comp_type

    output = os.path.join(output_dir, os.path.basename(spec.orig_file))
    prefix = spec.orig_base

    try:
        if repo.has_submodules() and with_submodules:
            repo.update_submodules()
            git_archive_submodules(repo, treeish, output, prefix,
                                   comp_type, comp_level, comp_opts)

        else:
            git_archive_single(treeish, output, prefix, comp_type, comp_level,
                               comp_opts, spec.orig_format, repo.path)
    except CommandExecFailed:
        gbp.log.err("Error generating submodules' archives")
        return False
    except OSError, err:
        gbp.log.err("Error creating %s: %s" % (output, err[0]))
        return False
    except GbpError:
        raise
    except Exception as e:
        gbp.log.err("Error creating %s: %s" % (output, e))
        return False
    return True


def prepare_upstream_tarball(repo, spec, options, output_dir):
    """
    Make sure we have an upstream tarball. This involves loooking in
    tarball_dir, symlinking or building it.
    """
    # look in tarball_dir first, if found force a symlink to it
    if options.tarball_dir:
        gbp.log.debug("Looking for orig tarball '%s' at '%s'" % (spec.orig_file, options.tarball_dir))
        if not rpm.RpmPkgPolicy.symlink_orig(spec.orig_file, options.tarball_dir, output_dir, force=True):
            gbp.log.info("Orig tarball '%s' not found at '%s'" % (spec.orig_file, options.tarball_dir))
        else:
            gbp.log.info("Orig tarball '%s' found at '%s'" % (spec.orig_file, options.tarball_dir))
    # build an orig unless the user forbids it, always build (and overwrite pre-existing) if user forces it
    if options.force_create or (not options.no_create_orig and not rpm.RpmPkgPolicy.has_orig(spec.orig_file, output_dir)):
        if not pristine_tar_build_orig(repo, spec.orig_file, output_dir, options):
            upstream_tree = git_archive_build_orig(repo, spec, output_dir, options)
            if options.pristine_tar_commit:
                if repo.pristine_tar.has_commit(spec.orig_file):
                    gbp.log.debug("%s already on pristine tar branch" %
                                  spec.orig_file)
                else:
                    archive = os.path.join(output_dir, spec.orig_file)
                    gbp.log.debug("Adding %s to pristine-tar branch" %
                                  archive)
                    repo.pristine_tar.commit(archive, upstream_tree)


def makedir(dir):
    output_dir = os.path.abspath(dir)

    try:
        os.mkdir(dir)
    except OSError, (e, msg):
        if e != errno.EEXIST:
            raise GbpError, "Cannot create dir %s" % dir
    return dir


def prepare_export_dir(dir):
    if not dir:
        dir = 'rpmbuild'
    return makedir(dir)


def pristine_tar_build_orig(repo, orig_file, output_dir, options):
    """
    build orig using pristine-tar
    @return: True: orig tarball build, False: noop
    """
    if options.pristine_tar:
        pt = PristineTar()
        if not repo.has_branch(pt.branch):
            gbp.log.warn('Pristine-tar branch "%s" not found' % pt.branch)
        pt.checkout(os.path.join(output_dir, orig_file))
        return True
    else:
        return False

def pristine_tar_build_orig(repo, orig_file, output_dir, options):
    """
    build orig using pristine-tar
    @return: True: orig tarball build, False: noop
    """
    if options.pristine_tar:
        if not repo.has_branch(repo.pristine_tar_branch):
            gbp.log.warn('Pristine-tar branch "%s" not found' %
                         repo.pristine_tar.branch)
        try:
            repo.pristine_tar.checkout(os.path.join(output_dir, orig_file))
            return True
        except CommandExecFailed:
            if options.pristine_tar_commit:
                gbp.log.debug("pristine-tar checkout failed, "
                              "will commit tarball due to "
                              "'--pristine-tar-commit'")
            else:
                raise
    return False

def get_upstream_tree(repo, spec, options):
    """Determine the upstream tree from the given options"""
    if options.upstream_tree.upper() == 'TAG':
        upstream_tree = repo.version_to_tag(options.upstream_tag, spec.version)
    elif options.upstream_tree.upper() == 'BRANCH':
        if not repo.has_branch(options.upstream_branch):
            raise GbpError("%s is not a valid branch" % options.upstream_branch)
        upstream_tree = options.upstream_branch
    else:
        upstream_tree = options.upstream_tree
    if not repo.has_treeish(upstream_tree):
        raise GbpError # git-ls-tree printed an error message already
    return upstream_tree


def git_archive_build_orig(repo, spec, output_dir, options):
    """
    Build orig tarball using git-archive

    @param repo: our git repository
    @type repo: L{RpmGitRepository}
    @param spec: spec file of the package
    @type spec: L{SpecFile}
    @param output_dir: where to put the tarball
    @type output_dir: C{Str}
    @param options: the parsed options
    @type options: C{dict} of options
    @return: the tree we built the tarball from
    @rtype: C{str}
    """
    upstream_tree = get_upstream_tree(repo, spec, options)
    gbp.log.info("%s does not exist, creating from '%s'" % (spec.orig_file,
                                                            upstream_tree))
    if options.comp_type:
        gbp.log.debug("Building upstream tarball with compression '%s -%s'" %
                      (options.comp_type, options.comp_level))
    if not git_archive(repo, spec, output_dir, upstream_tree,
                       options.comp_type,
                       options.comp_level,
                       options.with_submodules):
        raise GbpError("Cannot create upstream tarball at '%s'" % output_dir)
    return upstream_tree


def guess_comp_type(spec):
    if not spec.orig_comp:
        return ''
    if spec.orig_comp in compressor_opts:
        return spec.orig_comp
    if spec.orig_comp in compressor_aliases:
        return compressor_aliases[spec.orig_comp]
    raise GbpError("Unsupported compression extension in %s" % \
                   os.path.basename(spec.specfile))


def write_patch(patch, options):
    """Write the patch exported by 'git-format-patch' to it's final location
       (as specified in the commit)"""
    oldname = os.path.basename(patch)
    newname = oldname
    tmpname = patch + ".gbp"
    old = file(patch, 'r')
    tmp = file(tmpname, 'w')
    in_patch = False
    topic = None

    # Skip first line (from <sha1>)
    old.readline()
    for line in old:
        if in_patch:
            if line == '-- \n':
                # Found final signature, we're done:
                tmp.write(line)
                break
        else:
            if line.lower().startswith("gbp-pq-topic: "):
                topic = line.split(" ",1)[1].strip()
                gbp.log.debug("Topic %s found for %s" % (topic, patch))
                continue
            elif (line.startswith("diff --git a/") or
                  line.startswith("---")):
                in_patch = True
        tmp.write(line)

    tmp.close()
    old.close()

    if not options.patch_numbers:
        patch_re = re.compile("[0-9]+-(?P<name>.+)")
        m = patch_re.match(oldname)
        if m:
            newname = m.group('name')

    if topic:
        topicdir = os.path.join(options.packaging_dir, topic)
    else:
        topicdir = os.path.dirname(patch)

    if not os.path.isdir(topicdir):
        os.makedirs(topicdir, 0755)

    os.unlink(patch)
    dstname = os.path.join(topicdir, newname)
    gbp.log.debug("Moving %s to %s" % (tmpname, dstname))
    shutil.move(tmpname, dstname)

    return dstname


def gen_patches(repo, spec, totree, options):
    """Generate patches"""
    if options.upstream_tree.upper() == 'TAG':
        upstream_tree = repo.version_to_tag(options.upstream_tag, spec.version)
    elif options.upstream_tree.upper() == 'BRANCH':
        upstream_tree = options.upstream_branch
    else:
        raise GbpError, "Unknown value %s" % options.upstream_tree
    if not repo.has_treeish(upstream_tree):
        raise GbpError # git-ls-tree printed an error message already

    # Remove all old patches
    for n, p in spec.patches.iteritems():
        if p['autoupdate']:
            f = spec.specdir+"/"+p['filename']
            gbp.log.debug("Removing '%s'" % f)
            try:
                os.unlink(f)
            except OSError, (e, msg):
                if e != errno.ENOENT:
                    raise GbpError, "Failed to remove patch: %s" % msg
                else:
                    gbp.log.debug("%s does not exist." % f)

    gbp.log.info("Generating patches from git (%s..%s)" % (upstream_tree, totree))
    if repo.get_obj_type(totree) in ['tag', 'commit']:
        patches = repo.format_patches(upstream_tree, totree, spec.specdir)
    else:
        gbp.log.info("Repository object '%s' is neither tag nor commit, only generating a diff" % totree)
        diff = repo.diff(upstream_tree, totree)
        diff_filename = os.path.join(spec.specdir, '%s.diff' % totree)
        diff_file = open(diff_filename, 'w+')
        diff_file.writelines(diff)
        diff_file.close()
        patches = [diff_filename]

    filenames = []
    if patches:
        gbp.log.info("Regenerating patch series in '%s'." % spec.specdir)
        for patch in patches:
            filenames.append(os.path.basename(write_patch(patch, options)))

        spec.updatepatches(filenames)
    else:
        gbp.log.info("No changes - nothing to do.")


def is_native(repo, spec, options):
    try:
        get_upstream_tree(repo, spec, options)
    except GbpError:
        return True

    if repo.has_branch(options.upstream_branch):
        return False
    return True


def setup_builder(options, builder_args):
    """setup everything to use git-pbuilder"""
    if options.builder.startswith('rpmbuild'):
        builder_args.extend(['-ba',
                             '--define "_topdir %s"' % os.path.abspath(options.export_dir),
                             '--define "_builddir %%_topdir/%s"' % options.build_dir,
                             '--define "_rpmdir %%_topdir/%s"' % options.rpm_dir,
                             '--define "_sourcedir %%_topdir/%s"' % options.source_dir,
                             '--define "_specdir %%_topdir/%s"' % options.spec_dir,
                             '--define "_srpmdir %%_topdir/%s"' % options.srpm_dir,
                             '--define "_buildrootdir %%_topdir/%s"' % options.buildroot_dir])
    elif options.builder.startswith('osc'):
        builder_args.insert(0, 'build')
        options.source_dir = ''
        options.spec_dir = ''


def parse_args(argv, prefix):
    args = [ arg for arg in argv[1:] if arg.find('--%s' % prefix) == 0 ]
    builder_args = [ arg for arg in argv[1:] if arg.find('--%s' % prefix) == -1 ]

    # We handle these although they don't have a --git- prefix
    for arg in [ "--help", "-h", "--version" ]:
        if arg in builder_args:
            args.append(arg)

    try:
        parser = GbpOptionParserRpm(command=os.path.basename(argv[0]), prefix=prefix)
    except ConfigParser.ParsingError, err:
        gbp.log.err(err)
        return None, None, None

    tag_group = GbpOptionGroup(parser, "tag options", "options related to git tag creation")
    branch_group = GbpOptionGroup(parser, "branch options", "branch layout options")
    cmd_group = GbpOptionGroup(parser, "external command options", "how and when to invoke external commands and hooks")
    orig_group = GbpOptionGroup(parser, "orig tarball options", "options related to the creation of the orig tarball")
    export_group = GbpOptionGroup(parser, "export build-tree options", "alternative build tree related options")
    parser.add_option_group(tag_group)
    parser.add_option_group(orig_group)
    parser.add_option_group(branch_group)
    parser.add_option_group(cmd_group)
    parser.add_option_group(export_group)

    parser.add_boolean_config_file_option(option_name = "ignore-new", dest="ignore_new")
    parser.add_option("--git-verbose", action="store_true", dest="verbose", default=False,
                      help="verbose command execution")
    parser.add_config_file_option(option_name="color", dest="color", type='tristate')
    parser.add_config_file_option(option_name="notify", dest="notify", type='tristate')
    parser.add_config_file_option(option_name="vendor", action="store", dest="vendor")
    tag_group.add_option("--git-tag", action="store_true", dest="tag", default=False,
                      help="create a tag after a successful build")
    tag_group.add_option("--git-tag-only", action="store_true", dest="tag_only", default=False,
                      help="don't build, only tag and run the posttag hook")
    tag_group.add_option("--git-retag", action="store_true", dest="retag", default=False,
                      help="don't fail if the tag already exists")
    tag_group.add_boolean_config_file_option(option_name="sign-tags", dest="sign_tags")
    tag_group.add_config_file_option(option_name="keyid", dest="keyid")
    tag_group.add_config_file_option(option_name="packaging-tag", dest="packaging_tag")
    tag_group.add_config_file_option(option_name="upstream-tag", dest="upstream_tag")
    orig_group.add_config_file_option(option_name="upstream-tree", dest="upstream_tree")
    orig_group.add_boolean_config_file_option(option_name="pristine-tar", dest="pristine_tar")
    orig_group.add_boolean_config_file_option(option_name="pristine-tar-commit",
                                              dest="pristine_tar_commit")
    orig_group.add_config_file_option(option_name="force-create", dest="force_create",
                      help="force creation of upstream source tarball", action="store_true")
    orig_group.add_config_file_option(option_name="no-create-orig", dest="no_create_orig",
                      help="don't create upstream source tarball", action="store_true")
    orig_group.add_config_file_option(option_name="tarball-dir", dest="tarball_dir", type="path",
                      help="location to look for external tarballs")
    orig_group.add_config_file_option(option_name="compression-level", dest="comp_level",
                      help="Compression level, default is '%(compression-level)s'")
    branch_group.add_config_file_option(option_name="upstream-branch", dest="upstream_branch")
    branch_group.add_config_file_option(option_name="packaging-branch", dest="packaging_branch")
    branch_group.add_boolean_config_file_option(option_name = "ignore-branch", dest="ignore_branch")
    branch_group.add_boolean_config_file_option(option_name = "submodules", dest="with_submodules")
    cmd_group.add_config_file_option(option_name="builder", dest="builder",
                      help="command to build the package, default is '%(builder)s'")
    cmd_group.add_config_file_option(option_name="cleaner", dest="cleaner",
                      help="command to clean the working copy, default is '%(cleaner)s'")
    cmd_group.add_config_file_option(option_name="prebuild", dest="prebuild",
                      help="command to run before a build, default is '%(prebuild)s'")
    cmd_group.add_config_file_option(option_name="postexport", dest="postexport",
                      help="command to run after exporting the source tree, default is '%(postexport)s'")
    cmd_group.add_config_file_option(option_name="postbuild", dest="postbuild",
                      help="hook run after a successful build, default is '%(postbuild)s'")
    cmd_group.add_config_file_option(option_name="posttag", dest="posttag",
                      help="hook run after a successful tag operation, default is '%(posttag)s'")
    export_group.add_config_file_option(option_name="export-dir", dest="export_dir", type="path",
                      help="Build topdir, also export the sources under EXPORT_DIR, default is '%(export-dir)s'")
    export_group.add_config_file_option(option_name="rpmbuild-builddir", dest="build_dir", type="path",
                      help="subdir where package is built (under EXPORT_DIR), i.e. rpmbuild builddir, default is '%(rpmbuild-builddir)s'")
    export_group.add_config_file_option(option_name="rpmbuild-rpmdir", dest="rpm_dir", type="path",
                      help="subdir where ready binary packages are stored (under EXPORT_DIR), i.e. rpmbuild builddir, default is '%(rpmbuild-rpmdir)s'")
    export_group.add_config_file_option(option_name="rpmbuild-sourcedir", dest="source_dir", type="path",
                      help="subdir where package sources are stored (under EXPORT_DIR), i.e. rpmbuild sourcedir, default is '%(rpmbuild-sourcedir)s'")
    export_group.add_config_file_option(option_name="rpmbuild-specdir", dest="spec_dir", type="path",
                      help="subdir where package spec file is stored (under EXPORT_DIR), i.e. rpmbuild specdir, default is '%(rpmbuild-specdir)s'")
    export_group.add_config_file_option(option_name="rpmbuild-srpmdir", dest="srpm_dir", type="path",
                      help="subdir where ready sources package is stored (under EXPORT_DIR), i.e. rpmbuild srpmdir, default is '%(rpmbuild-srpmdir)s'")
    export_group.add_config_file_option(option_name="rpmbuild-buildrootdir", dest="buildroot_dir", type="path",
                      help="subdir for build-time alternative root (under EXPORT_DIR), i.e. rpmbuild buildrootdir, default is '%(rpmbuild-buildrootdir)s'")
    export_group.add_config_file_option("export", dest="export",
                      help="export treeish object TREEISH, default is '%(export)s'", metavar="TREEISH")
    export_group.add_option("--git-purge", action="store_true", dest="purge", default=False,
                      help="purge exported package build directory")
    export_group.add_config_file_option(option_name="packaging-dir",
                      dest="packaging_dir")
    export_group.add_config_file_option(option_name="specfile", dest="specfile")
    export_group.add_option("--git-export-only", action="store_true", dest="export_only", default=False,
                      help="only export packaging files, don't build")
    export_group.add_boolean_config_file_option("auto-patch-gen", dest="auto_patch_gen")
    export_group.add_boolean_config_file_option(option_name="patch-numbers", dest="patch_numbers")
    options, args = parser.parse_args(args)

    gbp.log.setup(options.color, options.verbose)
    if options.retag:
        if not options.tag and not options.tag_only:
            gbp.log.err("'--%sretag' needs either '--%stag' or '--%stag-only'" % (prefix, prefix, prefix))
            return None, None, None

    return options, args, builder_args


def main(argv):
    retval = 0
    prefix = "git-"
    cp = None

    options, gbp_args, builder_args = parse_args(argv, prefix)
    if not options:
        return 1

    try:
        repo = RpmGitRepository(os.path.curdir)
    except GitRepositoryError:
        gbp.log.err("%s is not a git repository" % (os.path.abspath('.')))
        return 1
    else:
        repo_dir = os.path.abspath(os.path.curdir)

    try:
        branch = repo.get_branch()
    except GitRepositoryError:
        branch = None

    try:
        if not options.export_only:
            Command(options.cleaner, shell=True)()
        if not options.ignore_new:
            (ret, out) = repo.is_clean()
            if not ret:
                gbp.log.err("You have uncommitted changes in your source tree:")
                gbp.log.err(out)
                raise GbpError, "Use --git-ignore-new to ignore."

        if not options.ignore_new and not options.ignore_branch:
            if branch != options.packaging_branch:
                gbp.log.err("You are not on branch '%s' but on '%s'" % (options.packaging_branch, branch))
                raise GbpError, "Use --git-ignore-branch to ignore or --git-packaging-branch to set the branch name."

        # Determine tree-ish to be exported
        if options.export == index_name:
            # Write a tree of the index
            tree = repo.write_tree()
        elif options.export in wc_names:
            # Write a tree of the working copy
            tree = write_wc(repo,
                            force=wc_names[options.export]['force'],
                            untracked=wc_names[options.export]['untracked'])
        else:
            tree = options.export
        if not repo.has_treeish(tree):
            raise GbpError # git-ls-tree printed an error message already

        # Dump from git to a temporary directory:
        tmp_dir = TempDir(dir=".")
        dump_dir = tmp_dir.path
        gbp.log.info("Dumping tree '%s' to '%s'" % (options.export, dump_dir))
        if not dump_tree(repo, dump_dir, tree, options.with_submodules):
            raise GbpError
        # Run postexport hook
        if options.postexport:
            RunAtCommand(options.postexport, shell=True,
                         extra_env={'GBP_GIT_DIR': repo.git_dir,
                                    'GBP_TMP_DIR': tmp_dir})(dir=tmp_dir)
        # Parse spec from dump dir to get version etc.
        if options.specfile:
            specfile = os.path.join(dump_dir, options.specfile)
            options.packaging_dir = os.path.dirname(specfile)
        else:
            options.packaging_dir, specfile = rpm.guess_spec(os.path.join(dump_dir, options.packaging_dir))
        gbp.log.debug("Using spec file '%s' from subdir '%s'" % (specfile, options.packaging_dir))
        spec = rpm.SpecFile(specfile)
        spec.debugprint()

        if not options.tag_only:
            # Setup builder opts
            setup_builder(options, builder_args)

            # Generate patches, if requested
            if options.auto_patch_gen:
                gen_patches(repo, spec, tree, options)

            # Prepare final export dirs
            export_dir = prepare_export_dir(options.export_dir)
            source_dir = makedir(os.path.join(export_dir, options.source_dir))
            spec_dir = makedir(os.path.join(export_dir, options.spec_dir))

            # Move packaging files
            gbp.log.info("Exporting packaging files in '%s' to '%s'" % (options.packaging_dir, export_dir))
            pkgfiles = os.listdir(options.packaging_dir)
            for f in pkgfiles:
                src = os.path.join(options.packaging_dir, f)
                if f == os.path.basename(specfile):
                    dst = os.path.join(spec_dir, f)
                    spec.specfile = os.path.abspath(dst)
                    spec.specdir = os.path.dirname(dst)
                else:
                    dst = os.path.join(source_dir, f)
                try:
                    if os.path.isdir(src):
                        # dir is not packaging files, skip it
                        continue
                    else:
                        shutil.copy2(src, dst)
                except IOError, err:
                    raise GbpError, "Error exporting files: %s" % err

            # Get/build the orig tarball
            options.comp_type = guess_comp_type(spec)
            if is_native(repo, spec, options):
                if spec.orig_file:
                    # Just build source tarball from the exported tree
                    gbp.log.debug("Building (native) src tarball with compression '%s -%s'" % (options.comp_type,
                                                                                                options.comp_level))
                    if not git_archive(repo, spec, source_dir, tree, options.comp_type,
                                       options.comp_level, options.with_submodules):
                        raise GbpError, "Cannot create source tarball at '%s'" % export_dir
            # Non-native packages: create orig tarball from upstream
            else:
                 prepare_upstream_tarball(repo, spec, options, source_dir)

            # Do actual build
            if not options.export_only and not options.tag_only:
                if options.prebuild:
                    RunAtCommand(options.prebuild, shell=True,
                                 extra_env={'GBP_GIT_DIR': repo.git_dir(),
                                            'GBP_BUILD_DIR': export_dir})(dir=export_dir)

                # Finally build the package:
                if options.builder.startswith("rpmbuild"):
                    builder_args.extend([spec.specfile])
                else:
                    builder_args.extend([os.path.basename(spec.specfile)])
                RunAtCommand(options.builder, builder_args, shell=True,
                             extra_env={'GBP_BUILD_DIR': export_dir})(dir=export_dir)
                if options.postbuild:
                    changes = os.path.abspath("%s/%s.changes" % (source_dir, spec.name))
                    gbp.log.debug("Looking for changes file %s" % changes)
                    Command(options.postbuild, shell=True,
                            extra_env={'GBP_CHANGES_FILE': changes,
                                       'GBP_BUILD_DIR': export_dir})()

        # Tag (note: tags the exported version)
        if options.tag or options.tag_only:
            gbp.log.info("Tagging %s" % spec.version)
            tag = repo.version_to_tag(options.packaging_tag, spec.version)
            if options.retag and repo.has_tag(tag):
                repo.delete_tag(tag)
            repo.create_tag(name=tag, msg="%s release %s" % (options.vendor, spec.version),
                            sign=options.sign_tags, keyid=options.keyid, commit=tree)
            if options.posttag:
                sha = repo.rev_parse("%s^{}" % tag)
                Command(options.posttag, shell=True,
                        extra_env={'GBP_TAG': tag,
                                   'GBP_BRANCH': branch,
                                   'GBP_SHA1': sha})()
    except CommandExecFailed:
        retval = 1
    except GbpError, err:
        if len(err.__str__()):
            gbp.log.err(err)
        retval = 1
    except rpm.NoSpecError, err:
        gbp.log.err("Failed determine spec file (%s)" % err)
        retval = 1
    finally:
        drop_index()

    if not options.tag_only:
        if options.purge and not retval and not options.export_only:
            RemoveTree(export_dir)()

        if cp and not gbp.notifications.notify(cp, not retval, options.notify):
            gbp.log.err("Failed to send notification")
            retval = 1

    return retval

if __name__ == '__main__':
    sys.exit(main(sys.argv))

# vim:et:ts=4:sw=4:et:sts=4:ai:set list listchars=tab\:»·,trail\:·:
