# vim: set fileencoding=utf-8 :
#
# (C) 2012 Intel Corporation <markus.lehtonen@linux.intel.com>
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""Test module for RPM command line tools of the git-buildpackage suite"""

import os
import shutil
import sys
import tempfile
from nose.plugins.skip import SkipTest
from StringIO import StringIO

import gbp.log
from gbp.git import GitRepository, GitRepositoryError

DATA_DIR = os.path.abspath(os.path.join('tests', 'TestRpmTools', 'testdata'))

class RpmTestGitRepository(GitRepository):
    """Git repository class for RPM tool tests"""

    def submodule_status(self):
        """
        Check current submodule status
        """
        out, err, ret = self._git_inout('submodule', ['status'],
                                        capture_stderr=True)
        if ret:
            raise GitRepositoryError("Cannot get submodule status: %s" %
                                     err.strip())
        submodules = {}
        for line in out.splitlines():
            module = line.strip()
            # Uninitialized
            status = module[0]
            if status == '-':
                sha1, path = module[1:].rsplit(' ', 1)
            else:
                commitpath = module[1:].rsplit(' ', 1)[0]
                sha1, path = commitpath.split(' ', 1)
            submodules[path] = (status, sha1)
        return submodules


def setup():
    """Module setup"""
    # Check if the testdata is up to date
    repo = RpmTestGitRepository('.')
    submodules = repo.submodule_status()
    status = submodules[os.path.join('tests', 'TestRpmTools', 'testdata')]
    if status[0] == '-':
        raise SkipTest("Skipping '%s', testdata directory not initialized. "\
                       "Consider doing 'git submodule update'" % __name__)


class TestRpmBase(object):
    """Base class for testing RPM cmdline tools of git-buildpackage"""

    @classmethod
    def setup_class(cls):
        """Test class case setup"""
        # Don't let git see that we're (possibly) under a git directory
        cls.orig_env = os.environ.copy()
        os.environ['GIT_CEILING_DIRECTORIES'] = os.getcwd()

    @classmethod
    def teardown_class(cls):
        """Test class case teardown"""
        # Return original environment
        os.environ = cls.orig_env

    def __init__(self):
        """Object initialization"""
        self._orig_dir = None
        self._tmpdir = None
        self._log = None
        self._loghandler = None

    def setup(self):
        """Test case setup"""
        # Change to a temporary directory
        self._orig_dir = os.getcwd()
        self._tmpdir = tempfile.mkdtemp(prefix='gbp_%s_' % __name__, dir='.')
        os.chdir(self._tmpdir)

        self._capture_log(True)

    def teardown(self):
        """Test case teardown"""
        # Restore original working dir
        os.chdir(self._orig_dir)
        shutil.rmtree(self._tmpdir)

        self._capture_log(False)

    @classmethod
    def _check_repo_state(cls, repo, current_branch, branches):
        """Check that repository is clean and given branches exist"""
        branch = repo.branch
        assert branch == current_branch
        assert repo.is_clean()
        assert set(repo.get_local_branches()) == set(branches)

    def _capture_log(self, capture=True):
        """ Capture log"""
        if capture and self._log is None:
            self._log = StringIO()
            self._loghandler = gbp.log.GbpStreamHandler(self._log, False)
            self._loghandler.addFilter(gbp.log.GbpFilter([gbp.log.WARNING,
                                                          gbp.log.ERROR]))
            gbp.log.LOGGER.addHandler(self._loghandler)
        elif self._log is not None:
            gbp.log.LOGGER.removeHandler(self._loghandler)
            self._loghandler = None
            self._log.close()
            self._log = None

    def _get_log(self):
        """Get the captured log output"""
        self._log.seek(0)
        return self._log.readlines()

    def _check_log(self, linenum, string):
        """Check that the specified line on log matches expectations"""
        if self._log is None:
            assert False, "BUG in unittests: no log captured!"
        output = self._get_log()[linenum].strip()
        assert output.startswith(string), ("Expected: '%s...' Got: '%s'" %
                                           (string, output))

    def _clear_log(self):
        """Clear the mock strerr"""
        if self._log is not None:
            self._log.seek(0)
            self._log.truncate()

# vim:et:ts=4:sw=4:et:sts=4:ai:set list listchars=tab\:»·,trail\:·:
