#!/usr/bin/env python
#
# This file is part of REPA: Release Engineering Process Assistant.
#
# Copyright (C) 2013 Intel Corporation
#
# REPA is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# version 2 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
# MA 02110-1301, USA.

"""
REPA: Release Engineering Process Assistant.

Copyright (C) Intel Corporation 2013
Licence: GPL version 2
Author: Ed Bartosh <eduard.bartosh@intel.com>

Common module.
Common functions, classes, exceptions.
"""

import sys
import os
import time
import json
import tempfile
import subprocess
import re
from functools import wraps, partial
from distutils.spawn import find_executable

from repa.jenkins import trigger_build

OBS_PROJECT_PREFIX = "home:prerelease:"

class RepaException(Exception):
    """Custom repa exception. All repa modules should use it."""
    pass

def get_prerelease(name, target):
    """Get name of prerelease project."""
    if name.startswith('submitgroup/'):
        name += '-group'
    return '%s%s:%s' % (OBS_PROJECT_PREFIX, target, name.replace('/', ':'))

def get_project_by_name(obs, name, target):
    """Lookup for a project in OBS by submission or group name."""
    if name.startswith('submitgroup/'):
        name += '-group'
    projects = list(obs.get_projects('^%s%s:%s$' % (OBS_PROJECT_PREFIX, target,
                                                    name.replace('/', ':'))))
    if not projects:
        raise RepaException('OBS project not found for %s %s' % \
                            (target, name))
    if len(projects) > 1:

        plist = '\n  '.join(prj for prj, _desc, _build_results in projects)
        raise RepaException('%s %s resolves into multiple projects:\n  %s' % \
                            (target, name, plist))

    return projects[0][0], json.loads(projects[0][1]), projects[0][2]


def _resolve_submissions(obs, name, target):
    """Get list of submissions with meta. Resolves submitgroups."""
    project, meta = get_project_by_name(obs, name, target)[:2]
    if name.startswith('submitgroup'):
        for subm in meta['submissions']:
            sprj, smeta = get_project_by_name(obs, subm, target)[:2]
            yield subm, sprj, smeta
    else:
        yield name, project, meta


def delete_project(obs, name, target):
    """Delete OBS project related to submission or submit group."""
    project = get_project_by_name(obs, name, target)[0]
    obs.delete_project(project)

def is_aggregate_package(obs, proj, pack):
    if "_aggregate" in obs.get_file_list(proj, pack):
        return True
    return False

def is_link_package(obs, proj, pack):
    if "_link" in obs.get_file_list(proj, pack):
        return True
    return False

def accept_or_reject(obs, submission, state, target, comment='',
                     jenkins_cred=None):
    """
    Create SRs and set their state for one submission or for a group.

    This can be done 2 ways - directly using OBS API and by triggering
                              jenkins job
    """
    for name, project, meta in _resolve_submissions(obs, submission, target):
        # osc submitreq [OPTIONS] SOURCEPRJ SOURCEPKG DESTPRJ [DESTPKG]
        # osc request accept [-m TEXT] ID
        print("submission %s" % str(name))

        submitter = meta.get('submitter')
        projects = '[' + ', '.join(meta['projects']) + ']'
        message = ''
        if submitter:
            message = "Submitter: %s\n" % submitter

        message += "Comments: %s \nGit project: %s\nTag: %s" \
            % (comment or "submission %s" % str(name),
               projects, meta['git_tag'])

        target_prj = str(meta['obs_target_prj'])

        if jenkins_cred:
            build, status, out = \
                trigger_build('re', {'action': state,
                                     'submission': str(name),
                                     'target_project': target_prj,
                                     'comment': comment}, jenkins_cred)
            print("Jenkins job: re, build #%s, status: %s" % (build, status))
            print(out)
        else:
            # Create SR
            org_source_packages=obs.get_source_packages(project)
            source_packages=[]
            for p in org_source_packages:
                if not is_aggregate_package(obs, project, p) and not is_link_package(obs, project, p):
                    source_packages.append(p)

            # If source_packages is not exists in target_prj
            # create a package in target_prj
            if state == 'accepted':
                org_target_packages=obs.get_package_list(target_prj)
                for package in source_packages:
                    if not package in org_target_packages:
                        obs.create_package(target_prj, package)

            reqid = obs.create_sr(project, source_packages,
                                  target_prj, message=message)

            print('created SR %s' % reqid)

            # and immediately set its state
            message = "SR %s is set to %s" % (reqid, state)
            if comment:
                message += comment
            obs.set_sr_state(reqid, state=state,
                         message=str(message), force=True)
            print('set SR state to', state)

    # delete submit group
    if submission.startswith('submitgroup'):
        delete_project(obs, submission, target)

class CancelRetryError(Exception):
    """Exception for handling cancelling of the retry loop. Needed for
    transparently re-raising the previous exception."""
    def __init__(self):
        Exception.__init__(self)
        self.typ, self.val, self.backtrace = sys.exc_info()

def retry(exceptions, tries=10, sleep=1):
    """Decorator for re-trying function calls"""
    def decorator(func):
        """The "real" decorator function"""
        @wraps(func)
        def wrap(*args, **kwargs):
            """Wrapper for re-trying func"""
            for attempt in range(1, tries + 1):
                try:
                    return func(*args, **kwargs)
                except CancelRetryError as err:
                    raise err.typ (err.val, err.backtrace)
                except exceptions as err:
                    if attempt >= tries:
                        raise
                    elif sleep:
                        time.sleep(sleep)
        return wrap
    return decorator

class Colorizer(object):
    """Colorize text with ANSI colors."""
    colors = {'black': '\033[90m', 'red':     '\033[91m',
              'green': '\033[92m', 'yellow':  '\033[93m',
              'blue':  '\033[94m', 'magenta': '\033[95m',
              'cyan':  '\033[96m', 'white':   '\033[97m'
             }
    reset = '\033[0m'

    def __init__(self, enable=True):
        self.enabled = enable

    def __getattr__(self, name):
        if name in self.colors:
            return partial(self.colorize, color=name)

    def disable(self):
        """Disable colorizing."""
        self.enabled = False

    def enable(self):
        """Enable colorizing."""
        self.enabled = True

    def colorize(self, text, color):
        """Colorize text."""
        if self.enabled:
            return self.reset + self.colors[color] + text + self.reset
        else:
            return text

def get_download_url(meta):
    """Get download url from meta."""
    if 'download_url' in meta:
        return meta['download_url']
    # Guess url from image url if download_url is not in the meta
    if 'images' not in meta:
        return
    for img in meta['images']:
        if 'url' in img:
            return img['url'].split('images')[0]


def get_obs_url(meta, buildurl='https://build.tizen.org'):
    """Get obs project url from meta."""
    if 'obs_url' in meta:
        return meta['obs_url']
    # Make it from git tag and obs_target_prj if obs_url is not in the meta
    if 'obs_target_prj' not in meta:
        return
    if 'name' not in meta and 'git_tag' not in meta:
        return
    name = meta.get('git_tag') or meta.get('name')
    return os.path.join(buildurl, 'project/show?project=home:prerelease:%s:%s'
                        % (meta['obs_target_prj'], name.replace('/', ':')))


def edit(content):
    """
    Launch an editor to get input from user.
    Returns: content of user input.
    """
    editor = os.getenv('EDITOR') or 'vi'

    if not find_executable(editor):
        raise RepaException("editor %s not found. Please set EDITOR "
                            "environment variable or install vi" % editor)

    fds, path = tempfile.mkstemp('.tmp', 'repa-', text=True)
    try:
        if content:
            os.write(fds, content)
        os.close(fds)

        try:
            subprocess.call([editor, path])
        except OSError as error:
            raise RepaException("Can't run %s %s: %s" % (editor, path, error))

        with open(path) as fobj:
            result = fobj.read()
    finally:
        os.unlink(path)

    return result

