
"""This module handles image related"""

import difflib
import glob
import os
from StringIO import StringIO

import requests

from .utils import JsonDict


def _get_file(url, suffix='.ks'):
    """Return (name, content) of ks file"""
    if url.startswith('http') or url.startswith('https'):
        response = requests.get(url + '/MD5SUMS')
        if response.status_code == 200:
            lines = response.content.split('\n')
            for line in lines:
                if suffix in line:
                    target = line.split('*')[-1]
                    f_response = requests.get(url + '/' + target)
                    if f_response.status_code == 200:
                        # use StringIO to keep wrapped lines
                        return target, StringIO(f_response.content).readlines()
                    break
    else:
        files = glob.glob(os.path.join(url, '*'+suffix))
        if files:
            only = files[0]
            try:
                with open(only, 'rb') as _only:
                    return only, _only.readlines()
            except IOError as err:
                print( err)

    return '', ''

def _parse_range(line):
    '''Start and Count'''
    def parse_start_count(chars):
        '''Count ommit when it's 1'''
        start, count = chars[1:].split(',')
        return int(start), int(count)

    _, delete, insert, _ = line.split()
    return {
        'delete': parse_start_count(delete),
        'insert': parse_start_count(insert),
        }

def _parse_hunks(arr, range_):
    '''Hunks'''
    count = 0
    hunks = []
    left_start = range_['delete'][0]
    right_start = range_['insert'][0]
    for line in arr:
        if line.startswith('+++') or line.startswith('---'):
            count += 1
            continue
        elif line.startswith(' '):
            type_ = 'context'
            hunk = {'type':'context',
                    'left_num':left_start,
                    'right_num':right_start,
                    'text':line[1:-1]}
            left_start += 1
            right_start += 1
        elif line.startswith('-'):
            type_ = 'delete'
            hunk = {}
            for right_line in arr[count:]:
                if right_line.startswith(' '):
                    hunk = {'type':'delete',
                            'left_num':left_start,
                            'left_text':line[1:-1]}
                    left_start += 1
                    break
                elif right_line.startswith('+'):
                    hunk = {'type':'change',
                            'left_num':left_start,
                            'right_num':right_start,
                            'left_text':line[1:-1],
                            'right_text':right_line[1:-1]}
                    arr.remove(right_line)
                    left_start += 1
                    right_start += 1
                    break
            if not hunk:
                hunk = {'type':'delete',
                        'left_num':left_start,
                        'left_text':line[1:-1]}
                left_start += 1
        elif line.startswith('+'):
            type_ = 'insert'
            hunk = {'type':type_,
                    'right_num':right_start,
                    'right_text':line[1:-1]}
            right_start += 1
        else:
            break
        hunks.append(hunk)
        count += 1
    return hunks

def _parse_sections(stream):
    """Parse ks diff sections"""
    sections = []
    stack = []
    head_stack = ''
    for line in stream:
        if not line.startswith('@@ '):
            stack.append(line)
            continue
        if not head_stack:
            head_stack = line
            continue
        range_ = _parse_range(head_stack)
        hunks = _parse_hunks(stack, range_)
        sections.append({
            'hunks': hunks
            })
        stack = []
        head_stack = line
    if stack and head_stack:
        range_ = _parse_range(head_stack)
        hunks = _parse_hunks(stack, range_)
        sections.append({
            'hunks': hunks
            })
    return sections

def ks_diff(old, new):
    """Return a ks file diff dict"""
    ks_old, ks_new = _get_file(old), _get_file(new)
    diff_list = []

    stream = difflib.unified_diff(ks_old[1], ks_new[1], fromfile=ks_old[0], \
            tofile=ks_new[0])
    for section in _parse_sections(stream):
        diff_list.append(section)
    return {'file_name': [ks_old[0], ks_new[0]], 'sections': diff_list}

def packages(url):
    """Return an image packages info dict"""
    content = _get_file(url, suffix='.packages')

    packages_info = JsonDict()

    for line in content[1]:
        if line.strip():
            name_arch, version, vcs = line.strip().split()
            name = name_arch.split('.')[0]
            setattr(packages_info, name, [])
            package_info = JsonDict()
            package_info.name = name
            package_info.version = JsonDict()
            ver, rel = version.split('-')
            package_info.version.epoch = 0 # we can't get epoch, so give 0.
            package_info.version.ver = ver
            package_info.version.rel = rel
            # maybe it occurs no vcs tag or vcs tag is 'None' not None in
            # rpm package, so give it a '#' to split it into ['', ''] later.
            package_info.version.vcs = ['#', vcs][vcs and vcs != 'None']
            packages_info[name].append(package_info)

    return packages_info
